# rush/main.py
import sys
import gi
import signal
import os

gi.require_version('Gtk', '4.0')
gi.require_version('Adw', '1')
from gi.repository import Gtk, Adw, Gio, GLib, Gdk

from rush.config import RESOURCES_DIR
from rush.utils.logger import logger
from rush.data.db import DatabaseManager
from rush.services.task_service import TaskService
from rush.services.board_service import BoardService
from rush.services.profile_service import ProfileService
from rush.services.flow_service import FlowService
from rush.ui.windows.main_window import MainWindow

class RushApp(Adw.Application):
    def __init__(self):
        super().__init__(application_id="com.manzoor.rush", flags=Gio.ApplicationFlags.FLAGS_NONE)
        self.db_manager = None
        self.main_window = None

    def do_startup(self):
        Adw.Application.do_startup(self)
        
        # Load CSS
        css_path = RESOURCES_DIR / "styles" / "style.css"
        if css_path.exists():
            provider = Gtk.CssProvider()
            provider.load_from_path(str(css_path))
            Gtk.StyleContext.add_provider_for_display(
                Gdk.Display.get_default(),
                provider,
                Gtk.STYLE_PROVIDER_PRIORITY_APPLICATION
            )
        else:
            logger.warning(f"Style.css not found at {css_path}")

    def do_activate(self):
        win = self.props.active_window
        if not win:
            # Initialize Database & Services
            self.db_manager = DatabaseManager()
            
            task_service = TaskService(self.db_manager)
            board_service = BoardService(self.db_manager)
            profile_service = ProfileService(self.db_manager)
            flow_service = FlowService(self.db_manager)
            
            # Apply Theme Preference
            self._apply_theme(profile_service)

            # Launch Window
            self.main_window = MainWindow(
                self, 
                task_service, 
                board_service, 
                profile_service, 
                flow_service
            )
            self.main_window.present()

    def _apply_theme(self, profile_service):
        prof = profile_service.get_user_profile()
        theme = prof.get("theme_pref", "system")
        manager = Adw.StyleManager.get_default()
        
        if theme == "dark":
            manager.set_color_scheme(Adw.ColorScheme.FORCE_DARK)
        elif theme == "light":
            manager.set_color_scheme(Adw.ColorScheme.FORCE_LIGHT)
        else:
            manager.set_color_scheme(Adw.ColorScheme.DEFAULT)

    def do_shutdown(self):
        logger.info("Shutting down...")
        if self.db_manager:
            self.db_manager.close()
        Adw.Application.do_shutdown(self)

def main():
    # Handle Ctrl+C
    signal.signal(signal.SIGINT, signal.SIG_DFL)
    app = RushApp()
    return app.run(sys.argv)

if __name__ == "__main__":
    sys.exit(main())
