# rush/data/db.py
import sqlite3
import os
import threading
from rush.config import USER_DATA_DIR
from rush.utils.logger import logger

CURRENT_DB_VERSION = 7

class DatabaseManager:
    DB_NAME = str(USER_DATA_DIR / "rush.db")

    def __init__(self):
        self._local = threading.local()
        if not USER_DATA_DIR.exists():
            try:
                USER_DATA_DIR.mkdir(parents=True, exist_ok=True)
            except OSError as e:
                logger.critical(f"Could not create data dir: {e}")
        self._ensure_db_init()

    def get_connection(self):
        if not hasattr(self._local, "conn") or self._local.conn is None:
            try:
                self._local.conn = sqlite3.connect(self.DB_NAME, timeout=20)
                self._local.conn.row_factory = sqlite3.Row
                self._local.conn.execute("PRAGMA journal_mode=WAL")
                self._local.conn.execute("PRAGMA foreign_keys = ON")
            except sqlite3.Error as e:
                logger.critical(f"DB Connect failed: {e}")
                raise e
        return self._local.conn

    def close(self):
        if hasattr(self._local, "conn") and self._local.conn:
            try:
                self._local.conn.close()
            except Exception as e:
                logger.error(f"Error closing DB: {e}")
            self._local.conn = None

    def _ensure_db_init(self):
        try:
            conn = self.get_connection()
            with conn:
                self._create_tables(conn)
                self._run_migrations(conn)
        except sqlite3.Error as e:
            logger.critical(f"Database init failed: {e}")

    def _create_tables(self, conn):
        conn.execute("CREATE TABLE IF NOT EXISTS schema_version (version INTEGER)")
        
        count = conn.execute("SELECT COUNT(*) FROM schema_version").fetchone()[0]
        if count == 0:
            conn.execute("INSERT INTO schema_version (version) VALUES (0)")

        conn.execute("""
            CREATE TABLE IF NOT EXISTS tasks (
                id TEXT PRIMARY KEY,
                title TEXT NOT NULL,
                description TEXT,
                status TEXT,
                task_type TEXT,
                priority INTEGER,
                due_date TEXT,
                recurrence TEXT,
                parent_recurrence_id TEXT,
                category_id TEXT,
                board_id TEXT,
                column_id TEXT,
                estimated_minutes INTEGER,
                actual_minutes INTEGER,
                created_at TEXT,
                completed_at TEXT,
                updated_at TEXT,
                checklist_json TEXT,
                mode TEXT DEFAULT 'Personal',
                tags TEXT DEFAULT '[]'
            )
        """)
        
        conn.execute("CREATE TABLE IF NOT EXISTS boards (id TEXT PRIMARY KEY, name TEXT, is_archived INTEGER DEFAULT 0)")
        
        conn.execute("""
            CREATE TABLE IF NOT EXISTS columns (
                id TEXT PRIMARY KEY,
                board_id TEXT,
                name TEXT,
                position INTEGER,
                color TEXT,
                wip_limit INTEGER,
                column_type TEXT DEFAULT 'wip',
                FOREIGN KEY(board_id) REFERENCES boards(id) ON DELETE CASCADE
            )
        """)

        conn.execute("""
            CREATE TABLE IF NOT EXISTS task_dependencies (
                parent_task_id TEXT,
                child_task_id TEXT,
                PRIMARY KEY (parent_task_id, child_task_id),
                FOREIGN KEY(parent_task_id) REFERENCES tasks(id) ON DELETE CASCADE,
                FOREIGN KEY(child_task_id) REFERENCES tasks(id) ON DELETE CASCADE
            )
        """)

        conn.execute("""
            CREATE TABLE IF NOT EXISTS task_attachments (
                id TEXT PRIMARY KEY,
                task_id TEXT,
                file_path TEXT,
                file_name TEXT,
                added_at TEXT DEFAULT CURRENT_TIMESTAMP,
                FOREIGN KEY(task_id) REFERENCES tasks(id) ON DELETE CASCADE
            )
        """)
        
        conn.execute("CREATE VIRTUAL TABLE IF NOT EXISTS tasks_fts USING fts5(title, description, content='tasks', content_rowid='rowid')")

        conn.execute("""
            CREATE TABLE IF NOT EXISTS user_profile (
                id INTEGER PRIMARY KEY CHECK (id = 1),
                name TEXT DEFAULT 'User',
                bio TEXT DEFAULT '',
                work_hours_start TEXT DEFAULT '09:00',
                work_hours_end TEXT DEFAULT '17:00',
                weekly_goal INTEGER DEFAULT 20,
                focus_pref_duration INTEGER DEFAULT 25,
                avatar_path TEXT,
                current_mode TEXT DEFAULT 'Personal',
                theme_pref TEXT DEFAULT 'system'
            )
        """)
        conn.execute("INSERT OR IGNORE INTO user_profile (id) VALUES (1)")
        
        conn.execute("""
            CREATE TABLE IF NOT EXISTS flow_sessions (
                id TEXT PRIMARY KEY,
                task_id TEXT,
                start_time TEXT,
                end_time TEXT,
                duration_seconds INTEGER,
                is_break INTEGER DEFAULT 0
            )
        """)

        self._create_triggers(conn)

    def _create_triggers(self, conn):
        conn.execute("DROP TRIGGER IF EXISTS tasks_ai")
        conn.execute("""
            CREATE TRIGGER tasks_ai AFTER INSERT ON tasks BEGIN
                INSERT INTO tasks_fts(rowid, title, description) VALUES (new.rowid, new.title, new.description);
            END;
        """)
        conn.execute("DROP TRIGGER IF EXISTS tasks_ad")
        conn.execute("""
            CREATE TRIGGER tasks_ad AFTER DELETE ON tasks BEGIN
                INSERT INTO tasks_fts(tasks_fts, rowid, title, description) VALUES('delete', old.rowid, old.title, old.description);
            END;
        """)
        conn.execute("DROP TRIGGER IF EXISTS tasks_au")
        conn.execute("""
            CREATE TRIGGER tasks_au AFTER UPDATE ON tasks BEGIN
                INSERT INTO tasks_fts(tasks_fts, rowid, title, description) VALUES('delete', old.rowid, old.title, old.description);
                INSERT INTO tasks_fts(rowid, title, description) VALUES (new.rowid, new.title, new.description);
            END;
        """)

    def _run_migrations(self, conn):
        cur = conn.execute("SELECT version FROM schema_version")
        row = cur.fetchone()
        version = row[0] if row else 0

        try:
            # Previous migrations (1-6)
            if version < 2:
                try: conn.execute("ALTER TABLE tasks ADD COLUMN parent_recurrence_id TEXT")
                except sqlite3.OperationalError: pass
                try: conn.execute("ALTER TABLE user_profile ADD COLUMN current_mode TEXT DEFAULT 'Personal'")
                except sqlite3.OperationalError: pass
            if version < 4:
                try: conn.execute("INSERT INTO tasks_fts(tasks_fts) VALUES('rebuild')")
                except sqlite3.OperationalError: pass
            if version < 5:
                try: conn.execute("ALTER TABLE tasks ADD COLUMN mode TEXT DEFAULT 'Personal'")
                except sqlite3.OperationalError: pass
            if version < 6:
                try: conn.execute("ALTER TABLE user_profile ADD COLUMN theme_pref TEXT DEFAULT 'system'")
                except sqlite3.OperationalError: pass
            
            # NEW: Migration 7 (Tags)
            if version < 7:
                logger.info("Migrating DB to v7 (Adding Tags)...")
                try: 
                    conn.execute("ALTER TABLE tasks ADD COLUMN tags TEXT DEFAULT '[]'")
                except sqlite3.OperationalError as e: 
                    logger.warning(f"Migration warning: {e}")

            if version != CURRENT_DB_VERSION:
                conn.execute("UPDATE schema_version SET version = ?", (CURRENT_DB_VERSION,))
                logger.info(f"Database schema updated to version: {CURRENT_DB_VERSION}")
        except sqlite3.Error as e:
            logger.error(f"Migration failed: {e}")

    def drop_all(self):
        self.close()
        if os.path.exists(self.DB_NAME):
            try:
                os.remove(self.DB_NAME)
            except OSError as e:
                logger.error(f"Failed to delete DB: {e}")
        self.__init__()
