# rush/services/board_service.py
import sqlite3
import uuid
from rush.services.base_service import BaseService
from rush.data.models import Board, Column
from rush.utils.logger import logger
from rush.utils.event_bus import EventBus

class BoardService(BaseService):
    def __init__(self, db_manager):
        super().__init__(db_manager)
        self.bus = EventBus.get_default()

    def get_all_boards(self):
        conn = self.db.get_connection()
        try:
            # 1. Fetch Boards
            b_rows = conn.execute("SELECT * FROM boards WHERE is_archived = 0").fetchall()
            boards = []
            
            for br in b_rows:
                # 2. Fetch Columns for each board
                c_rows = conn.execute("SELECT * FROM columns WHERE board_id = ? ORDER BY position", (br['id'],)).fetchall()
                cols = []
                for cr in c_rows:
                    cols.append(Column(
                        id=cr['id'],
                        name=cr['name'],
                        position=cr['position'],
                        color=cr['color'],
                        wip_limit=cr['wip_limit']
                    ))
                
                boards.append(Board(
                    id=br['id'],
                    name=br['name'],
                    columns=cols
                ))
            
            # Create Default Board if none exist
            if not boards:
                count = conn.execute("SELECT COUNT(*) FROM boards").fetchone()[0]
                if count == 0:
                    def_id = str(uuid.uuid4())
                    self.create_board("Main Board", def_id)
                    return self.get_all_boards()
                
            return boards
        except sqlite3.Error as e:
            logger.error(f"Fetch boards error: {e}")
            return []

    def create_board(self, name, board_id=None):
        if not board_id: board_id = str(uuid.uuid4())
        conn = self.db.get_connection()
        try:
            with conn:
                conn.execute("INSERT INTO boards (id, name) VALUES (?, ?)", (board_id, name))
                # Add Default Columns
                cols = ["To Do", "In Progress", "Done"]
                for i, cname in enumerate(cols):
                    self.add_column(board_id, cname, i, emit=False)
            
            logger.info(f"Board created: {name}")
            self.bus.emit('board-changed')
        except sqlite3.Error as e:
            logger.error(f"Create board error: {e}")

    def delete_board(self, board_id):
        """
        Deletes a board and its columns. 
        Tasks are NOT deleted; they are unlinked (moved to backlog).
        """
        conn = self.db.get_connection()
        try:
            with conn:
                # 1. Unlink tasks
                conn.execute("""
                    UPDATE tasks 
                    SET board_id = NULL, column_id = NULL 
                    WHERE board_id = ?
                """, (board_id,))
                
                # 2. Delete columns (Cascade handling varies by DB config, manual is safer here)
                conn.execute("DELETE FROM columns WHERE board_id = ?", (board_id,))
                
                # 3. Delete board
                conn.execute("DELETE FROM boards WHERE id = ?", (board_id,))
            
            logger.info(f"Board deleted: {board_id}")
            self.bus.emit('board-changed')
            self.bus.emit('task-updated')
        except sqlite3.Error as e:
            logger.error(f"Delete board error: {e}")

    def add_column(self, board_id, name, position=None, emit=True):
        conn = self.db.get_connection()
        try:
            if position is None:
                row = conn.execute("SELECT MAX(position) as m FROM columns WHERE board_id = ?", (board_id,)).fetchone()
                position = (row['m'] + 1) if row and row['m'] is not None else 0
            
            cid = str(uuid.uuid4())
            with conn:
                conn.execute("INSERT INTO columns (id, board_id, name, position) VALUES (?, ?, ?, ?)", 
                             (cid, board_id, name, position))
            
            if emit:
                self.bus.emit('board-changed')
        except sqlite3.Error:
            pass

    def update_column(self, col_id, name):
        conn = self.db.get_connection()
        try:
            with conn:
                conn.execute("UPDATE columns SET name = ? WHERE id = ?", (name, col_id))
            self.bus.emit('board-changed')
        except sqlite3.Error: pass

    def delete_column(self, col_id):
        conn = self.db.get_connection()
        try:
            with conn:
                # Unlink tasks in this column
                conn.execute("UPDATE tasks SET column_id = NULL WHERE column_id = ?", (col_id,))
                conn.execute("DELETE FROM columns WHERE id = ?", (col_id,))
            self.bus.emit('board-changed')
            self.bus.emit('task-updated')
        except sqlite3.Error: pass
