# rush/services/export_service.py
import json
import os
import sqlite3
from datetime import datetime
from rush.utils.logger import logger
from rush.services.report_service import ReportService
from rush.data.models import Task, Status, Priority, ChecklistItem

class ExportService:
    def __init__(self, task_service, profile_service, board_service):
        self.task_service = task_service
        self.profile_service = profile_service
        self.board_service = board_service
        self.report_service = ReportService(task_service)
        self.db = task_service.db

    def export_json(self, file_path):
        try:
            conn = self.db.get_connection()
            sessions = [dict(r) for r in conn.execute("SELECT * FROM flow_sessions").fetchall()]
            dependencies = [dict(r) for r in conn.execute("SELECT * FROM task_dependencies").fetchall()]
            
            data = {
                "metadata": {
                    "exported_at": datetime.now().isoformat(), 
                    "version": "1.1.0", 
                    "app": "Rush"
                },
                "profile": self.profile_service.get_user_profile(),
                "boards": self._serialize_boards(),
                "tasks": [self._serialize_task(t) for t in self.task_service.get_all_tasks()],
                "dependencies": dependencies,
                "flow_sessions": sessions
            }
            
            with open(file_path, 'w') as f:
                json.dump(data, f, indent=2)
            logger.info(f"Exported JSON to {file_path}")
        except Exception as e:
            logger.error(f"JSON Export failed: {e}")
            raise e

    def import_json(self, file_path):
        """Restores application data from a JSON backup."""
        try:
            with open(file_path, 'r') as f:
                data = json.load(f)
            
            if not isinstance(data, dict):
                raise ValueError("Invalid backup file format")

            conn = self.db.get_connection()
            with conn:
                # 1. Profile
                p = data.get("profile", {})
                if p:
                    self.profile_service.update_profile(
                        p.get("name", "User"), p.get("bio", ""), 
                        "09:00", "17:00", 
                        p.get("weekly_goal", 20), None
                    )

                # 2. Boards & Columns
                # Note: We use INSERT OR REPLACE/IGNORE strategies to avoid conflicts
                for b in data.get("boards", []):
                    conn.execute("INSERT OR REPLACE INTO boards (id, name, is_archived) VALUES (?, ?, 0)", (b['id'], b['name']))
                    for col in b.get("columns", []):
                        conn.execute("""
                            INSERT OR REPLACE INTO columns (id, board_id, name, position, color, wip_limit) 
                            VALUES (?, ?, ?, ?, ?, ?)
                        """, (col['id'], b['id'], col['name'], col['position'], col['color'], col.get('wip_limit', 0)))

                # 3. Tasks
                for item in data.get("tasks", []):
                    self._restore_single_task(item, conn)

                # 4. Dependencies
                for dep in data.get("dependencies", []):
                    try:
                        conn.execute("INSERT OR IGNORE INTO task_dependencies (parent_task_id, child_task_id) VALUES (?, ?)", 
                                     (dep['parent_task_id'], dep['child_task_id']))
                    except sqlite3.Error: pass

                # 5. Flow Sessions
                for sess in data.get("flow_sessions", []):
                    conn.execute("INSERT OR IGNORE INTO flow_sessions (id, task_id, start_time, duration_seconds) VALUES (?, ?, ?, ?)", 
                                 (sess['id'], sess['task_id'], sess['start_time'], sess['duration_seconds']))

            logger.info("Import completed successfully.")
            return True
        except Exception as e:
            logger.error(f"Import failed: {e}")
            raise e

    def reset_application_data(self):
        """Destructive: Wipes all user data."""
        self.db.drop_all()
        # Re-initialize basic tables
        self.db._ensure_db_init()

    def _restore_single_task(self, item, conn):
        try:
            due = item.get('due_date') # Keep as string or convert if DB expects string
            created = item.get('created_at', datetime.now().isoformat())
            updated = item.get('updated_at', datetime.now().isoformat())
            completed = item.get('completed_at')

            # Reconstruct checklist JSON
            checklist_data = item.get('checklist', [])
            checklist_json = json.dumps(checklist_data)

            # Map status/priority back to values if they are stored as names
            status = item.get('status', 'TODO')
            if status == "IN_PROGRESS": status = "in_progress"
            elif status == "DONE": status = "done"
            else: status = "todo"

            params = (
                item['id'], item['title'], item.get('description', ''),
                status, "todo", int(item.get('priority', 1)),
                due, item.get('recurrence'), None, # parent_recurrence_id
                item.get('category_id'), item.get('board_id'), item.get('column_id'),
                item.get('estimated_minutes', 0), item.get('actual_minutes', 0),
                created, updated, completed, checklist_json, item.get('mode', 'Personal')
            )
            
            conn.execute("""
                INSERT OR REPLACE INTO tasks (
                    id, title, description, status, task_type, priority, due_date, 
                    recurrence, parent_recurrence_id, category_id, board_id, column_id, estimated_minutes, 
                    actual_minutes, created_at, updated_at, completed_at, checklist_json, mode
                ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
            """, params)
        except Exception as e:
            logger.error(f"Failed to restore task {item.get('id')}: {e}")

    # ... [Existing PDF and Serialization methods remain unchanged] ...
    def export_advanced_pdf(self, file_path):
        try:
            from weasyprint import HTML
        except ImportError:
            logger.error("WeasyPrint not installed.")
            raise ImportError("Please install 'weasyprint' to enable PDF export.")

        profile = self.profile_service.get_user_profile()
        stats = self.report_service.get_extended_stats(days=30)
        tasks = self.task_service.get_all_tasks()
        boards = self.board_service.get_all_boards()
        
        conn = self.task_service.db.get_connection()
        deps = conn.execute("""
            SELECT p.title as parent, c.title as child 
            FROM task_dependencies td
            JOIN tasks p ON td.parent_task_id = p.id
            JOIN tasks c ON td.child_task_id = c.id
        """).fetchall()

        pending = [t for t in tasks if t.status.name != 'DONE']
        completed = [t for t in tasks if t.status.name == 'DONE']
        
        html_content = self._generate_html_template(profile, stats, boards, pending, completed, deps)
        HTML(string=html_content).write_pdf(file_path)

    def _generate_html_template(self, profile, stats, boards, pending, completed, deps):
        return f"""
        <html>
        <head>
            <style>
                body {{ font-family: sans-serif; padding: 40px; }}
                h1 {{ color: #3584e4; }}
                table {{ width: 100%; border-collapse: collapse; margin-top: 20px; }}
                th {{ background: #eee; padding: 10px; text-align: left; }}
                td {{ border-bottom: 1px solid #ddd; padding: 10px; }}
            </style>
        </head>
        <body>
            <h1>Rush Report: {profile.get('name', 'User')}</h1>
            <p><strong>Exported:</strong> {datetime.now().strftime('%Y-%m-%d')}</p>
            <h2>Pending Tasks</h2>
            <table>
                <thead><tr><th>Task</th><th>Priority</th><th>Due</th></tr></thead>
                <tbody>
                    {''.join(f"<tr><td>{t.title}</td><td>{t.priority.name}</td><td>{t.due_date or '-'}</td></tr>" for t in pending)}
                </tbody>
            </table>
        </body>
        </html>
        """

    def _serialize_boards(self):
        boards = self.board_service.get_all_boards()
        out = []
        for b in boards:
            b_dict = {"id": b.id, "name": b.name, "columns": []}
            for c in b.columns:
                ctype = c.column_type.value if hasattr(c.column_type, 'value') else "wip"
                b_dict["columns"].append({
                    "id": c.id, "name": c.name, "position": c.position,
                    "color": c.color, "column_type": ctype,
                    "wip_limit": c.wip_limit
                })
            out.append(b_dict)
        return out

    def _serialize_task(self, t):
        return {
            "id": t.id,
            "title": t.title,
            "description": t.description,
            "status": t.status.name if hasattr(t.status, 'name') else "TODO",
            "priority": t.priority.value if hasattr(t.priority, 'value') else 1,
            "due_date": t.due_date.isoformat() if t.due_date else None,
            "created_at": t.created_at.isoformat(),
            "completed_at": t.completed_at.isoformat() if t.completed_at else None,
            "board_id": t.board_id,
            "column_id": t.column_id,
            "estimated_minutes": t.estimated_minutes,
            "actual_minutes": t.actual_minutes,
            "tags": t.tags,
            "recurrence": t.recurrence,
            "checklist": [{'text': i.text, 'completed': i.completed} for i in t.checklist],
            "mode": t.mode
        }
