# rush/services/flow_service.py
import sqlite3
import uuid
from datetime import datetime
from rush.services.base_service import BaseService
from rush.utils.logger import logger

class FlowService(BaseService):
    def __init__(self, db_manager):
        super().__init__(db_manager)

    def start_session(self, task_id, expected_duration_minutes):
        session_id = str(uuid.uuid4())
        start_time = datetime.now().isoformat()
        
        # We initially record the expected duration. 
        # If stopped early, update_session_duration should be called.
        conn = self.db.get_connection()
        try:
            with conn:
                conn.execute("""
                    INSERT INTO flow_sessions (id, task_id, start_time, duration_seconds, is_break)
                    VALUES (?, ?, ?, ?, 0)
                """, (session_id, task_id, start_time, expected_duration_minutes * 60))
            logger.info(f"Flow Started: {expected_duration_minutes}m (ID: {session_id})")
            return session_id
        except sqlite3.Error as e:
            logger.error(f"Start flow error: {e}")
            return None

    def complete_session(self, session_id, actual_duration_seconds):
        """Called when a timer finishes or is stopped manually."""
        if not session_id: return
        
        end_time = datetime.now().isoformat()
        conn = self.db.get_connection()
        try:
            with conn:
                conn.execute("""
                    UPDATE flow_sessions 
                    SET end_time = ?, duration_seconds = ?
                    WHERE id = ?
                """, (end_time, actual_duration_seconds, session_id))
            logger.info(f"Flow Finished: {actual_duration_seconds}s")
        except sqlite3.Error as e:
            logger.error(f"Complete flow error: {e}")

    def get_today_stats(self):
        # Return total minutes focused today
        today = datetime.now().strftime("%Y-%m-%d")
        conn = self.db.get_connection()
        try:
            rows = conn.execute("""
                SELECT duration_seconds FROM flow_sessions 
                WHERE start_time LIKE ?
            """, (f"{today}%",)).fetchall()
            
            total_seconds = sum(r['duration_seconds'] for r in rows)
            return int(total_seconds / 60)
        except sqlite3.Error as e:
            logger.error(f"Get today stats error: {e}")
            return 0
