import sqlite3
from rush.services.base_service import BaseService
from rush.utils.logger import logger
from rush.utils.event_bus import EventBus

class ProfileService(BaseService):
    def __init__(self, db_manager):
        super().__init__(db_manager)
        self.bus = EventBus.get_default()

    def get_user_profile(self):
        conn = self.db.get_connection()
        try:
            row = conn.execute("SELECT * FROM user_profile WHERE id = 1").fetchone()
            if row:
                return dict(row)
        except sqlite3.Error as e:
            logger.error(f"Fetch profile error: {e}")
        return {"name": "User", "bio": "", "current_mode": "Personal", "theme_pref": "system"}

    def update_profile(self, name, bio, work_start, work_end, weekly_goal, avatar_path):
        conn = self.db.get_connection()
        try:
            with conn:
                conn.execute("""
                    UPDATE user_profile 
                    SET name = ?, bio = ?, work_hours_start = ?, work_hours_end = ?, 
                        weekly_goal = ?, avatar_path = ?
                    WHERE id = 1
                """, (name, bio, work_start, work_end, weekly_goal, avatar_path))
            
            self.bus.emit('profile-updated', ['name', 'bio', 'avatar'])
        except sqlite3.Error as e:
            logger.error(f"Update profile error: {e}")

    def update_mode(self, mode):
        conn = self.db.get_connection()
        try:
            with conn:
                conn.execute("UPDATE user_profile SET current_mode = ? WHERE id = 1", (mode,))
            
            logger.info(f"Switched mode to: {mode}")
            self.bus.emit('profile-updated', ['current_mode'])
            self.bus.emit('task-updated') 
        except sqlite3.Error as e:
            logger.error(f"Update mode error: {e}")

    def update_theme_pref(self, theme):
        conn = self.db.get_connection()
        try:
            with conn:
                conn.execute("UPDATE user_profile SET theme_pref = ? WHERE id = 1", (theme,))
            logger.info(f"Theme preference saved: {theme}")
        except sqlite3.Error as e:
            logger.error(f"Update theme error: {e}")

    def get_current_mode(self):
        profile = self.get_user_profile()
        return profile.get('current_mode', 'Personal')

    def get_available_modes(self):
        return ["Personal", "Work", "Study", "Creative"]
