# rush/services/report_service.py
import sqlite3
from datetime import datetime, timedelta, date
from rush.services.base_service import BaseService
from rush.utils.logger import logger

class ReportService(BaseService):
    def __init__(self, task_service):
        super().__init__(task_service.db)
        self.task_service = task_service

    def get_extended_stats(self, days=30):
        """Calculates meaningful derived metrics using optimized SQL queries."""
        conn = self.db.get_connection()
        stats = {
            "completion_rate": 0,
            "consistency_score": 0,
            "avg_session_len": 0,
            "total_done": 0,
            "total_hours": 0.0
        }
        
        try:
            start_date = (datetime.now() - timedelta(days=days)).isoformat()
            
            # 1. Completion Rate: (Done / (Done + Todo)) * 100 for the period
            counts = conn.execute("""
                SELECT 
                    SUM(CASE WHEN status = 'done' THEN 1 ELSE 0 END) as done,
                    SUM(CASE WHEN status != 'done' THEN 1 ELSE 0 END) as pending
                FROM tasks 
                WHERE (completed_at >= ? OR created_at >= ?)
            """, (start_date, start_date)).fetchone()
            
            done = counts['done'] or 0
            pending = counts['pending'] or 0
            
            if (done + pending) > 0:
                stats["completion_rate"] = int((done / (done + pending)) * 100)
            stats["total_done"] = done

            # 2. Focus Consistency & Average Session
            # Aggregates sessions to find daily output variance
            daily_focus = conn.execute("""
                SELECT date(start_time) as d, SUM(duration_seconds) as secs, COUNT(*) as sessions
                FROM flow_sessions
                WHERE start_time >= ?
                GROUP BY d
            """, (start_date,)).fetchall()
            
            if daily_focus:
                hours_list = [r['secs'] / 3600 for r in daily_focus]
                total_secs = sum(r['secs'] for r in daily_focus)
                total_sessions = sum(r['sessions'] for r in daily_focus)
                
                stats["total_hours"] = round(total_secs / 3600, 1)
                
                if total_sessions > 0:
                    stats["avg_session_len"] = round((total_secs / total_sessions) / 60, 1)
                
                # Consistency calculation
                # We need to account for days with 0 hours which aren't in the GROUP BY result
                # but technically affect consistency.
                # For simplicity in this local app, we calculate variance only on active days
                # but penalize the score based on ratio of active days vs total days.
                
                if len(hours_list) > 1:
                    avg_h = sum(hours_list) / len(hours_list)
                    variance = sum((x - avg_h) ** 2 for x in hours_list) / len(hours_list)
                    std_dev = variance ** 0.5
                    
                    # Score 0-100: lower deviation is better.
                    # We also multiply by activity_ratio (active_days / days) to penalize inactivity.
                    consistency_base = max(0, min(100, int(100 - (std_dev * 20))))
                    activity_ratio = len(hours_list) / days
                    stats["consistency_score"] = int(consistency_base * activity_ratio)
                else:
                    # Not enough data for variance
                    stats["consistency_score"] = 50 if stats["total_hours"] > 0 else 0

        except sqlite3.Error as e:
            logger.error(f"Extended stats error: {e}")
        
        return stats

    def get_completion_trend(self, days=14):
        """Aggregates daily completion counts for trend charts."""
        conn = self.db.get_connection()
        trend = []
        try:
            # Safe date calculation
            start_date_dt = date.today() - timedelta(days=days)
            start_date_str = start_date_dt.isoformat()
            
            rows = conn.execute("""
                SELECT date(completed_at) as d, COUNT(*) as c 
                FROM tasks 
                WHERE status='done' AND date(completed_at) >= ?
                GROUP BY d
            """, (start_date_str,)).fetchall()
            
            counts = {r['d']: r['c'] for r in rows if r['d']}
            
            # Ensure all days are represented, even if 0
            for i in range(days-1, -1, -1):
                d = date.today() - timedelta(days=i)
                d_str = d.strftime("%Y-%m-%d")
                trend.append((d_str, counts.get(d_str, 0)))
                
        except Exception as e:
            logger.error(f"Trend error: {e}")
        return trend

    def get_time_distribution(self, days=7):
        """Aggregates focus time by board for pie charts."""
        conn = self.db.get_connection()
        data = {}
        try:
            start_date = (datetime.now() - timedelta(days=days)).isoformat()
            sql = """
                SELECT b.name, SUM(fs.duration_seconds) as secs
                FROM flow_sessions fs
                JOIN tasks t ON fs.task_id = t.id
                JOIN boards b ON t.board_id = b.id
                WHERE fs.start_time >= ?
                GROUP BY b.name
            """
            rows = conn.execute(sql, (start_date,)).fetchall()
            for r in rows:
                if r['secs'] and r['secs'] > 0:
                    data[r['name']] = round(r['secs'] / 3600, 1)
        except Exception as e: 
            logger.error(f"Dist error: {e}")
        return data
