# rush/services/task_service.py
import json
import sqlite3
import uuid
import os
import shutil
from datetime import datetime, date
from rush.services.base_service import BaseService
from rush.data.models import Task, Status, Priority, TaskType, ChecklistItem
import rush.data.queries as queries
from rush.utils.logger import logger
from rush.config import USER_DATA_DIR

class TaskService(BaseService):
    def get_all_tasks(self):
        conn = self.db.get_connection()
        try:
            rows = conn.execute(queries.SELECT_ALL_TASKS).fetchall()
            return [t for t in (self._row_to_task(r) for r in rows) if t is not None]
        except sqlite3.Error as e:
            logger.error(f"Fetch tasks error: {e}")
            return []

    def search_tasks(self, query):
        if not query: return []
        safe_query = query.replace('"', '""') 
        fts_query = f'"{safe_query}" *' 
        like_query = f"%{query}%"
        conn = self.db.get_connection()
        try:
            rows = conn.execute(queries.SEARCH_TASKS, (fts_query, like_query)).fetchall()
            return [t for t in (self._row_to_task(r) for r in rows) if t is not None]
        except sqlite3.Error: return []

    def get_all_dependencies(self):
        conn = self.db.get_connection()
        try:
            rows = conn.execute("SELECT parent_task_id, child_task_id FROM task_dependencies").fetchall()
            return [(r['parent_task_id'], r['child_task_id']) for r in rows]
        except sqlite3.Error: return []

    def create_task(self, task: Task):
        conn = self.db.get_connection()
        try:
            checklist_json = json.dumps([{"text": i.text, "completed": i.completed} for i in task.checklist])
            tags_json = json.dumps(task.tags) # Serialize tags
            
            params = (
                task.id, task.title, task.description, task.status.value, task.task_type.value,
                task.priority.value, task.due_date.isoformat() if task.due_date else None,
                task.recurrence, None, task.category_id, task.board_id, task.column_id,
                task.estimated_minutes, task.actual_minutes,
                task.created_at.isoformat(), datetime.now().isoformat(),
                task.completed_at.isoformat() if task.completed_at else None,
                checklist_json, task.mode, tags_json
            )
            with conn:
                conn.execute(queries.INSERT_TASK, params)
            logger.info(f"Task created: {task.title}")
        except sqlite3.Error as e:
            logger.error(f"Create task error: {e}")

    def update_task(self, task: Task):
        conn = self.db.get_connection()
        try:
            checklist_json = json.dumps([{"text": i.text, "completed": i.completed} for i in task.checklist])
            tags_json = json.dumps(task.tags) # Serialize tags
            
            params = (
                task.title, task.description, task.status.value, task.priority.value,
                task.due_date.isoformat() if task.due_date else None,
                task.recurrence, task.board_id, task.column_id,
                task.estimated_minutes, task.actual_minutes,
                task.completed_at.isoformat() if task.completed_at else None,
                datetime.now().isoformat(), checklist_json, task.mode, tags_json,
                task.id
            )
            with conn:
                conn.execute(queries.UPDATE_TASK, params)
            logger.info(f"Task updated: {task.title}")
        except sqlite3.Error as e:
            logger.error(f"Update task error: {e}")

    def delete_task(self, task_id):
        conn = self.db.get_connection()
        try:
            with conn:
                conn.execute(queries.DELETE_TASK, (task_id,))
            logger.info(f"Task deleted: {task_id}")
        except sqlite3.Error: pass

    def get_kanban_board_tasks(self, board_id):
        all_tasks = self.get_all_tasks()
        board_tasks = [t for t in all_tasks if t.board_id == board_id]
        grouped = {}
        for t in board_tasks:
            col = t.column_id or "uncategorized"
            if col not in grouped: grouped[col] = []
            grouped[col].append(t)
        return grouped

    def get_blockers(self, task_id):
        conn = self.db.get_connection()
        try:
            rows = conn.execute(queries.SELECT_BLOCKERS, (task_id,)).fetchall()
            return [t for t in (self._row_to_task(r) for r in rows) if t is not None]
        except sqlite3.Error: return []

    def get_blocking(self, task_id):
        conn = self.db.get_connection()
        try:
            rows = conn.execute(queries.SELECT_BLOCKING, (task_id,)).fetchall()
            return [t for t in (self._row_to_task(r) for r in rows) if t is not None]
        except sqlite3.Error: return []

    def add_dependency(self, parent_id, child_id):
        conn = self.db.get_connection()
        try:
            with conn:
                conn.execute(queries.INSERT_DEPENDENCY, (parent_id, child_id))
        except sqlite3.Error: pass

    def remove_dependency(self, parent_id, child_id):
        conn = self.db.get_connection()
        try:
            with conn:
                conn.execute(queries.DELETE_DEPENDENCY, (parent_id, child_id))
        except sqlite3.Error: pass

    def move_task_to_column(self, task_id, column_id):
        conn = self.db.get_connection()
        try:
            with conn:
                conn.execute("UPDATE tasks SET column_id = ? WHERE id = ?", (column_id, task_id))
        except sqlite3.Error: pass

    # --- ATTACHMENTS ---
    def get_attachments(self, task_id):
        conn = self.db.get_connection()
        try:
            rows = conn.execute(queries.SELECT_ATTACHMENTS, (task_id,)).fetchall()
            return [dict(r) for r in rows]
        except sqlite3.Error: return []

    def add_attachment(self, task_id, source_path):
        try:
            filename = os.path.basename(source_path)
            att_id = str(uuid.uuid4())
            att_dir = USER_DATA_DIR / "attachments"
            att_dir.mkdir(parents=True, exist_ok=True)
            dest_path = att_dir / f"{att_id}_{filename}"
            shutil.copy2(source_path, dest_path)
            
            conn = self.db.get_connection()
            with conn:
                conn.execute("INSERT INTO task_attachments (id, task_id, file_path, file_name) VALUES (?, ?, ?, ?)",
                             (att_id, task_id, str(dest_path), filename))
            return True
        except Exception as e:
            logger.error(f"Add attachment error: {e}")
            return False

    def delete_attachment(self, att_id):
        conn = self.db.get_connection()
        try:
            row = conn.execute("SELECT file_path FROM task_attachments WHERE id = ?", (att_id,)).fetchone()
            if row and row['file_path'] and os.path.exists(row['file_path']):
                try: os.remove(row['file_path'])
                except OSError: pass
            with conn:
                conn.execute(queries.DELETE_ATTACHMENT, (att_id,))
        except sqlite3.Error: pass

    def _row_to_task(self, row):
        try:
            due = date.fromisoformat(row['due_date']) if row['due_date'] else None
            created = datetime.fromisoformat(row['created_at']) if row['created_at'] else datetime.now()
            completed = datetime.fromisoformat(row['completed_at']) if row['completed_at'] else None
            
            checklist = []
            if row['checklist_json']:
                try:
                    data = json.loads(row['checklist_json'])
                    checklist = [ChecklistItem(x['text'], x['completed']) for x in data]
                except json.JSONDecodeError: pass

            tags = []
            if 'tags' in row.keys() and row['tags']:
                try: tags = json.loads(row['tags'])
                except json.JSONDecodeError: pass

            status_val = row['status']
            if status_val == "done": status = Status.DONE
            elif status_val == "in_progress": status = Status.IN_PROGRESS
            else: status = Status.TODO

            return Task(
                id=row['id'],
                title=row['title'],
                description=row['description'],
                status=status,
                priority=Priority(row['priority'] if row['priority'] else 1),
                task_type=TaskType.TODO,
                due_date=due,
                recurrence=row['recurrence'],
                board_id=row['board_id'],
                column_id=row['column_id'],
                estimated_minutes=row['estimated_minutes'] or 0,
                actual_minutes=row['actual_minutes'] or 0,
                checklist=checklist,
                created_at=created,
                completed_at=completed,
                mode=row['mode'] if 'mode' in row.keys() and row['mode'] else "Personal",
                tags=tags
            )
        except Exception as e:
            return None
