# rush/ui/dialogs/export_dialog.py
import gi
import os
from datetime import datetime
gi.require_version('Gtk', '4.0')
gi.require_version('Adw', '1')
from gi.repository import Gtk, Adw, Gio, GLib

from rush.services.export_service import ExportService
from rush.services.report_service import ReportService

class ExportDialog(Adw.Window):
    def __init__(self, parent, task_service, profile_service=None, board_service=None):
        super().__init__(transient_for=parent, modal=True)
        self.task_service = task_service
        self.profile_service = profile_service
        self.board_service = board_service
        self.set_title("Export Data")
        self.set_default_size(500, 400)
        
        self.export_service = ExportService(task_service, profile_service, board_service)
        self.report_service = ReportService(task_service)
        
        self._setup_ui()

    def _setup_ui(self):
        self.toast_overlay = Adw.ToastOverlay()
        self.set_content(self.toast_overlay)

        content = Adw.PreferencesPage()
        self.toast_overlay.set_child(content)
        
        group_json = Adw.PreferencesGroup(title="Backup")
        content.add(group_json)
        
        row_json = Adw.ActionRow(title="Export to JSON", subtitle="Full database dump for backup.")
        row_json.set_icon_name("application-json-symbolic")
        btn_json = Gtk.Button(label="Export")
        btn_json.connect("clicked", self._on_export_json)
        row_json.add_suffix(btn_json)
        group_json.add(row_json)

        group_report = Adw.PreferencesGroup(title="Reports")
        content.add(group_report)
        
        row_html = Adw.ActionRow(title="Weekly Review", subtitle="Generate a printable HTML summary.")
        row_html.set_icon_name("x-office-document-symbolic")
        btn_html = Gtk.Button(label="Generate")
        btn_html.add_css_class("suggested-action")
        btn_html.connect("clicked", self._on_export_report)
        row_html.add_suffix(btn_html)
        group_report.add(row_html)
        
        row_pdf = Adw.ActionRow(title="Project Report (PDF)", subtitle="Full task list for printing.")
        row_pdf.set_icon_name("x-office-document-symbolic")
        btn_pdf = Gtk.Button(label="Export PDF")
        btn_pdf.connect("clicked", self._on_export_pdf)
        row_pdf.add_suffix(btn_pdf)
        group_report.add(row_pdf)

    def _on_export_json(self, btn):
        filename = "rush_export.json"
        path = os.path.abspath(filename)
        self._run_export(self.export_service.export_json, path)

    def _on_export_report(self, btn):
        filename = f"Rush_Weekly_Report_{datetime.now().strftime('%Y-%m-%d')}.html"
        path = os.path.abspath(filename)
        self._run_export(self.report_service.generate_weekly_report, path)

    def _on_export_pdf(self, btn):
        filename = f"Rush_Report_{datetime.now().strftime('%Y-%m-%d')}.pdf"
        path = os.path.join(os.path.expanduser("~/Documents"), filename)
        self._run_export(self.export_service.export_pdf, path)

    def _run_export(self, func, path):
        def task():
            func(path)
            return path
        
        def on_done(result_path):
            self._show_open_notification(result_path)
            
        self.task_service.run_in_thread(on_done, task)

    def _show_open_notification(self, path):
        toast = Adw.Toast.new(f"Saved to {os.path.basename(path)}")
        toast.set_button_label("Open")
        toast.connect("button-clicked", lambda t: self._open_file(path))
        self.toast_overlay.add_toast(toast)

    def _open_file(self, path):
        f = Gio.File.new_for_path(path)
        Gio.AppInfo.launch_default_for_uri(f.get_uri(), None)
