# rush/ui/dialogs/search_dialog.py
import gi
gi.require_version('Gtk', '4.0')
gi.require_version('Adw', '1')
from gi.repository import Gtk, Adw, Gio, Gdk, GLib
from rush.ui.dialogs.task_editor import TaskEditorDialog
from rush.ui.objects import TaskObject

class SearchDialog(Adw.Window):
    def __init__(self, parent, task_service, app_window):
        super().__init__(transient_for=parent, modal=True)
        self.task_service = task_service
        self.app_window = app_window
        self.set_default_size(600, 500)
        self.set_title("Jump to...")
        
        controller = Gtk.EventControllerKey()
        controller.connect("key-pressed", self._on_key_pressed)
        self.add_controller(controller)
        
        self._setup_ui()

    def _setup_ui(self):
        toolbar_view = Adw.ToolbarView()
        self.set_content(toolbar_view)
        
        header = Adw.HeaderBar()
        toolbar_view.add_top_bar(header)
        
        main_box = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=12)
        main_box.set_margin_top(12); main_box.set_margin_bottom(12)
        main_box.set_margin_start(12); main_box.set_margin_end(12)
        
        self.search_entry = Gtk.SearchEntry()
        self.search_entry.set_placeholder_text("Type to search tasks...")
        self.search_entry.set_hexpand(True)
        self.search_entry.connect("search-changed", self._on_search_changed)
        self.search_entry.connect("activate", self._on_activate)
        main_box.append(self.search_entry)
        
        self.model = Gio.ListStore(item_type=TaskObject)
        self.selection = Gtk.SingleSelection(model=self.model)
        
        factory = Gtk.SignalListItemFactory()
        factory.connect("setup", self._on_setup_row)
        factory.connect("bind", self._on_bind_row)
        
        self.list_view = Gtk.ListView(model=self.selection, factory=factory)
        self.list_view.add_css_class("navigation-sidebar")
        self.list_view.set_single_click_activate(True)
        self.list_view.connect("activate", self._on_list_activate)
        
        scrolled = Gtk.ScrolledWindow()
        scrolled.set_child(self.list_view)
        scrolled.set_vexpand(True)
        scrolled.add_css_class("card") 
        main_box.append(scrolled)
        
        toolbar_view.set_content(main_box)
        
        self._populate_initial_data()

    def _populate_initial_data(self):
        self._add_commands()
        self.task_service.run_in_thread(self._on_recent_tasks_loaded, self.task_service.get_all_tasks)

    def _on_recent_tasks_loaded(self, tasks):
        GLib.idle_add(self._append_recent_tasks, tasks)

    def _append_recent_tasks(self, tasks):
        tasks.sort(key=lambda t: t.updated_at, reverse=True)
        for t in tasks[:15]: 
            obj = TaskObject(t)
            self.model.append(obj)

    def _add_commands(self, filter_text=""):
        # Safe access to app_window methods
        commands = [
            ("New Task", "Create a new task", lambda: self.app_window._on_quick_add(None, None)),
            ("Go to Dashboard", "View Dashboard", lambda: self.app_window.stack.set_visible_child_name("dashboard")),
            ("Go to Kanban", "View Board", lambda: self.app_window.stack.set_visible_child_name("kanban")),
            ("Go to Flow", "Focus Timer", lambda: self.app_window.stack.set_visible_child_name("flow")),
            ("Toggle Theme", "Switch Light/Dark", lambda: self._toggle_theme()),
        ]
        
        for cmd, desc, action in commands:
            if not filter_text or filter_text.lower() in cmd.lower():
                t = TaskObject(None) 
                t.is_command = True
                t.cmd_title = cmd
                t.cmd_desc = desc
                t.cmd_action = action
                self.model.append(t)

    def _on_search_changed(self, entry):
        query = entry.get_text()
        self.model.remove_all()
        
        if not query: 
            self._populate_initial_data()
            return
        
        if query.startswith(">"):
            self._add_commands(query[1:].strip())
        else:
            self.task_service.run_in_thread(self._on_search_results, self.task_service.search_tasks, query)

    def _on_search_results(self, results):
        GLib.idle_add(self._update_model_safely, results)

    def _update_model_safely(self, results):
        if not results: return
        for t in results:
            obj = TaskObject(t)
            self.model.append(obj)

    def _on_activate(self, entry):
        item = self.selection.get_selected_item()
        if item: self._execute_item(item)

    def _on_list_activate(self, listview, position):
        item = self.model.get_item(position)
        if item: self._execute_item(item)

    def _execute_item(self, item):
        self.close()
        if item.is_command and item.cmd_action:
            item.cmd_action()
        elif item.task:
            win = self.app_window
            dialog = TaskEditorDialog(
                win, 
                win.task_service, 
                win.board_service, 
                win.profile_service, 
                item.task
            )
            dialog.present()

    def _toggle_theme(self):
        manager = Adw.StyleManager.get_default()
        is_dark = manager.get_dark()
        manager.set_color_scheme(Adw.ColorScheme.FORCE_LIGHT if is_dark else Adw.ColorScheme.FORCE_DARK)
        # Note: This is a temporary toggle. To save persistent state, 
        # the user should use the Settings dialog.

    def _on_setup_row(self, factory, item):
        box = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=2)
        box.set_margin_start(12); box.set_margin_end(12)
        box.set_margin_top(8); box.set_margin_bottom(8)
        lbl_main = Gtk.Label(xalign=0, css_classes=["heading"])
        lbl_sub = Gtk.Label(xalign=0, css_classes=["caption", "dim-label"])
        box.append(lbl_main)
        box.append(lbl_sub)
        item.set_child(box)

    def _on_bind_row(self, factory, item):
        widget = item.get_child()
        obj = item.get_item()
        lbl_main = widget.get_first_child()
        lbl_sub = lbl_main.get_next_sibling()
        
        if obj.is_command:
            lbl_main.set_text(obj.cmd_title)
            lbl_sub.set_text(obj.cmd_desc)
            widget.remove_css_class("task-row")
            widget.add_css_class("command-row")
        elif obj.task:
            lbl_main.set_text(obj.task.title)
            lbl_sub.set_text(f"{obj.task.status.name} • {obj.task.priority.name}")
            widget.remove_css_class("command-row")
            widget.add_css_class("task-row")

    def _on_key_pressed(self, controller, keyval, keycode, state):
        if keyval == Gdk.KEY_Escape:
            self.close()
            return True
        return False
