# rush/ui/dialogs/settings.py
import gi
import os
import json
import webbrowser
from datetime import datetime
from pathlib import Path
gi.require_version('Gtk', '4.0')
gi.require_version('Adw', '1')
from gi.repository import Gtk, Adw, Gio, GLib, Gdk

from rush.config import RESOURCES_DIR
from rush.services.export_service import ExportService
from rush.utils.event_bus import EventBus

class SettingsDialog(Adw.PreferencesWindow):
    def __init__(self, parent, task_service, profile_service, board_service):
        super().__init__(transient_for=parent, modal=True)
        self.task_service = task_service
        self.profile_service = profile_service
        self.board_service = board_service
        self.bus = EventBus.get_default()
        
        self.export_service = ExportService(task_service, profile_service, board_service)
        self.style_manager = Adw.StyleManager.get_default()
        
        self.set_title("Settings")
        self.set_default_size(550, 750)
        
        self.current_avatar_path = None
        
        self.add(self._setup_profile_page())
        self.add(self._setup_data_page())
        self.add(self._setup_about_page())
        
        self._load_data()

    def _setup_profile_page(self):
        page = Adw.PreferencesPage(title="Profile", icon_name="user-info-symbolic")
        
        # --- IDENTITY ---
        group_bio = Adw.PreferencesGroup(title="Identity")
        self.entry_name = Adw.EntryRow(title="Name")
        self.entry_bio = Adw.EntryRow(title="Bio")
        group_bio.add(self.entry_name)
        group_bio.add(self.entry_bio)
        page.add(group_bio)
        
        # --- APPEARANCE ---
        group_av = Adw.PreferencesGroup(title="Appearance")
        
        # Theme
        self.row_theme = Adw.ComboRow(title="Theme")
        self.row_theme.set_model(Gtk.StringList.new(["System", "Light", "Dark"]))
        self.row_theme.connect("notify::selected", self._on_theme_changed)
        group_av.add(self.row_theme)
        
        # Avatar
        self.row_avatar = Adw.ActionRow(title="User Avatar", subtitle="Select a profile picture")
        
        box_av = Gtk.Box(spacing=12)
        box_av.set_valign(Gtk.Align.CENTER)
        
        self.img_avatar_preview = Adw.Avatar(size=48, show_initials=True)
        box_av.append(self.img_avatar_preview)
        
        btn_av = Gtk.Button(label="Choose Image")
        btn_av.set_valign(Gtk.Align.CENTER)
        btn_av.connect("clicked", self._on_choose_avatar)
        box_av.append(btn_av)
        
        self.row_avatar.add_suffix(box_av)
        group_av.add(self.row_avatar)
        
        page.add(group_av)
        
        # --- PRODUCTIVITY ---
        group_goals = Adw.PreferencesGroup(title="Productivity")
        self.row_goal = Adw.SpinRow.new_with_range(0, 100, 1)
        self.row_goal.set_title("Weekly Task Goal")
        group_goals.add(self.row_goal)
        page.add(group_goals)

        return page

    def _setup_data_page(self):
        page = Adw.PreferencesPage(title="Data", icon_name="drive-harddisk-symbolic")
        
        # --- EXPORT ---
        group_export = Adw.PreferencesGroup(title="Export")
        row_pdf = Adw.ActionRow(title="Export PDF Report", subtitle="Advanced statistics and list")
        btn_pdf = Gtk.Button(icon_name="x-office-document-symbolic", valign=Gtk.Align.CENTER)
        btn_pdf.connect("clicked", self._on_export_advanced_pdf)
        row_pdf.add_suffix(btn_pdf)
        group_export.add(row_pdf)

        row_json = Adw.ActionRow(title="Export JSON Backup", subtitle="Full app state (Portable)")
        btn_json = Gtk.Button(icon_name="document-save-symbolic", valign=Gtk.Align.CENTER)
        btn_json.connect("clicked", self._on_export_json)
        row_json.add_suffix(btn_json)
        group_export.add(row_json)
        page.add(group_export)

        # --- RESTORE ---
        group_restore = Adw.PreferencesGroup(title="Restore")
        row_import = Adw.ActionRow(title="Import Backup", subtitle="Restore from .json file")
        btn_import = Gtk.Button(label="Import", valign=Gtk.Align.CENTER)
        btn_import.add_css_class("suggested-action")
        btn_import.connect("clicked", self._on_import_clicked)
        row_import.add_suffix(btn_import)
        group_restore.add(row_import)
        page.add(group_restore)
        
        # --- DANGER ZONE ---
        group_danger = Adw.PreferencesGroup(title="Danger Zone")
        
        row_reset = Adw.ActionRow(title="Reset Application", subtitle="Wipe all data and start fresh")
        btn_reset = Gtk.Button(label="Reset All", css_classes=["destructive-action"])
        btn_reset.connect("clicked", self._on_reset_clicked)
        row_reset.add_suffix(btn_reset)
        group_danger.add(row_reset)
        
        page.add(group_danger)
        return page

    def _setup_about_page(self):
        page = Adw.PreferencesPage(title="About", icon_name="help-about-symbolic")
        group = Adw.PreferencesGroup()
        
        # --- APP HEADER ---
        box = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=12, margin_top=24, margin_bottom=24)
        box.set_halign(Gtk.Align.CENTER)
        
        icon_img = Gtk.Image(pixel_size=128)
        
        # Robust Icon Finding Logic
        # 1. Check rush/resources/assets/icon.svg (Installed/Packaged)
        # 2. Check ../assets/icon.svg (Local Dev Root)
        possible_paths = [
            RESOURCES_DIR / "assets" / "icon.svg",
            RESOURCES_DIR.parent.parent / "assets" / "icon.svg",
            Path("assets/icon.svg").resolve() 
        ]
        
        icon_found = False
        for p in possible_paths:
            if p.exists():
                try:
                    icon_img.set_from_file(str(p))
                    icon_found = True
                    break
                except: pass
        
        if not icon_found:
            icon_img.set_from_icon_name("application-x-executable")
            
        box.append(icon_img)
        box.append(Gtk.Label(label="Rush", css_classes=["display-1"]))
        box.append(Gtk.Label(label="Version 1.1.0", css_classes=["title-2"]))
        box.append(Gtk.Label(label="Created by Manzoor Ahmed", css_classes=["dim-label"]))
        
        group.add(Gtk.Frame(child=box))
        page.add(group)
        
        # --- LINKS ---
        link_group = Adw.PreferencesGroup(title="Links")
        links = [
            ("Donate", "https://rush.manzoorahmed.com/donate", "emblem-favorite-symbolic"),
            ("Website", "https://rush.manzoorahmed.com", "network-wired-symbolic"),
            ("License (GPLv3)", "https://www.gnu.org/licenses/gpl-3.0.html", "text-x-generic-symbolic")
        ]
        
        for title, url, icon in links:
            row = Adw.ActionRow(title=title, subtitle=url)
            row.set_icon_name(icon)
            row.set_activatable(True)
            row.connect("activated", lambda x, u=url: webbrowser.open(u))
            link_group.add(row)
            
        page.add(link_group)
        
        return page

    def _load_data(self):
        profile = self.profile_service.get_user_profile()
        self.entry_name.set_text(profile.get('name', ''))
        self.entry_bio.set_text(profile.get('bio', ''))
        self.row_goal.set_value(profile.get('weekly_goal', 20))
        
        self.current_avatar_path = profile.get('avatar_path')
        self._update_avatar_preview()
        
        theme = profile.get("theme_pref", "system")
        idx = 1 if theme == "light" else 2 if theme == "dark" else 0
        self.row_theme.set_selected(idx)
        
        self.entry_name.connect("apply", self._save_profile)
        self.entry_bio.connect("apply", self._save_profile)
        self.row_goal.connect("notify::value", self._save_profile)

    def _on_theme_changed(self, row, param):
        idx = row.get_selected()
        val = "system" if idx == 0 else "light" if idx == 1 else "dark"
        self.profile_service.update_theme_pref(val)
        
        manager = Adw.StyleManager.get_default()
        if val == "dark": manager.set_color_scheme(Adw.ColorScheme.FORCE_DARK)
        elif val == "light": manager.set_color_scheme(Adw.ColorScheme.FORCE_LIGHT)
        else: manager.set_color_scheme(Adw.ColorScheme.DEFAULT)

    def _on_choose_avatar(self, btn):
        dialog = Gtk.FileChooserNative(title="Select Avatar", transient_for=self, action=Gtk.FileChooserAction.OPEN)
        
        filter_img = Gtk.FileFilter()
        filter_img.set_name("Images")
        filter_img.add_mime_type("image/png")
        filter_img.add_mime_type("image/jpeg")
        filter_img.add_mime_type("image/svg+xml")
        dialog.add_filter(filter_img)
        
        def on_response(d, response):
            if response == Gtk.ResponseType.ACCEPT:
                f = d.get_file()
                if f:
                    self.current_avatar_path = f.get_path()
                    self._update_avatar_preview()
                    self._save_profile()
            d.destroy()
            
        dialog.connect("response", on_response)
        dialog.show()

    def _update_avatar_preview(self):
        name = self.entry_name.get_text()
        self.img_avatar_preview.set_text(name if name else "U")
        if self.current_avatar_path and os.path.exists(self.current_avatar_path):
            try:
                texture = Gdk.Texture.new_from_filename(self.current_avatar_path)
                self.img_avatar_preview.set_custom_image(texture)
            except: 
                self.img_avatar_preview.set_custom_image(None)

    def _save_profile(self, *args):
        self.profile_service.update_profile(
            self.entry_name.get_text(), 
            self.entry_bio.get_text(), 
            "09:00", "17:00", 
            int(self.row_goal.get_value()), 
            self.current_avatar_path
        )

    def _on_export_advanced_pdf(self, btn):
        self._file_chooser_save("Rush_Report.pdf", self.export_service.export_advanced_pdf)

    def _on_export_json(self, btn):
        self._file_chooser_save("Rush_Backup.json", self.export_service.export_json)

    def _file_chooser_save(self, default_name, callback):
        dialog = Gtk.FileChooserNative(title="Save File", transient_for=self, action=Gtk.FileChooserAction.SAVE)
        dialog.set_current_name(f"{datetime.now().strftime('%Y%m%d')}_{default_name}")
        
        def on_response(d, res):
            if res == Gtk.ResponseType.ACCEPT:
                try:
                    callback(d.get_file().get_path())
                    self.add_toast(Adw.Toast.new("Export successful"))
                except Exception as e:
                    self.add_toast(Adw.Toast.new(f"Export failed: {e}"))
            d.destroy()
        
        dialog.connect("response", on_response)
        dialog.show()

    def _on_import_clicked(self, btn):
        dialog = Gtk.FileChooserNative(title="Import Backup", transient_for=self, action=Gtk.FileChooserAction.OPEN)
        filter_json = Gtk.FileFilter()
        filter_json.set_name("JSON Files")
        filter_json.add_pattern("*.json")
        dialog.add_filter(filter_json)
        
        def on_response(d, res):
            if res == Gtk.ResponseType.ACCEPT:
                path = d.get_file().get_path()
                try:
                    self.export_service.import_json(path)
                    self.add_toast(Adw.Toast.new("Data imported successfully!"))
                    # FIX: Emit with empty list as argument
                    self.bus.emit('task-updated')
                    self.bus.emit('board-changed')
                    self.bus.emit('profile-updated', [])
                except Exception as e:
                    self.add_toast(Adw.Toast.new(f"Import failed: {e}"))
            d.destroy()
            
        dialog.connect("response", on_response)
        dialog.show()

    def _on_reset_clicked(self, btn):
        dialog = Adw.MessageDialog(
            heading="Reset Application?",
            body="This will delete ALL tasks, boards, and settings. This action cannot be undone.",
            transient_for=self
        )
        dialog.add_response("cancel", "Cancel")
        dialog.add_response("reset", "RESET EVERYTHING")
        dialog.set_response_appearance("reset", Adw.ResponseAppearance.DESTRUCTIVE)
        
        def on_response(d, res):
            if res == "reset":
                self.export_service.reset_application_data()
                self.add_toast(Adw.Toast.new("Application reset complete."))
                # FIX: Emit with empty list as argument to prevent TypeError
                self.bus.emit('task-updated')
                self.bus.emit('board-changed')
                self.bus.emit('profile-updated', [])
                self.close()
            d.close()
            
        dialog.connect("response", on_response)
        dialog.present()
