# rush/ui/views/calendar_view.py
import gi
from datetime import datetime, date
gi.require_version('Gtk', '4.0')
gi.require_version('Adw', '1')
from gi.repository import Gtk, Adw, GLib

from rush.ui.widgets.task_row import TaskRowWidget
from rush.ui.widgets.empty_state import EmptyStateWidget
from rush.ui.dialogs.task_editor import TaskEditorDialog
from rush.utils.event_bus import EventBus

class CalendarView(Adw.Bin):
    __gtype_name__ = 'CalendarView'

    def __init__(self, task_service, profile_service, board_service):
        super().__init__()
        self.task_service = task_service
        self.profile_service = profile_service
        self.board_service = board_service
        self.bus = EventBus.get_default()
        
        self.tasks_cache = []
        self.selected_date = date.today()
        
        self._setup_ui()
        
        self.bus.connect('task-updated', self._refresh)
        self._refresh()

    def _setup_ui(self):
        split = Adw.OverlaySplitView()
        self.set_child(split)
        
        # Sidebar: Calendar
        page_cal = Adw.NavigationPage(title="Calendar", tag="cal")
        box_cal = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=12)
        box_cal.set_margin_top(12); box_cal.set_margin_start(12); box_cal.set_margin_end(12)
        
        self.calendar = Gtk.Calendar()
        self.calendar.connect("day-selected", self._on_date_changed)
        # Card styling for calendar widget
        self.calendar.add_css_class("card")
        box_cal.append(self.calendar)
        
        lbl = Gtk.Label(label="Dots indicate due tasks", css_classes=["caption", "dim-label"])
        box_cal.append(lbl)
        
        page_cal.set_child(box_cal)
        split.set_sidebar(page_cal)
        split.set_sidebar_width_fraction(0.35)
        
        # Content: Task List
        page_list = Adw.NavigationPage(title="Tasks", tag="list")
        self.list_box = Gtk.ListBox()
        # Use new vibrant styling
        self.list_box.add_css_class("boxed-list")
        self.list_box.set_selection_mode(Gtk.SelectionMode.NONE)
        self.list_box.connect("row-activated", self._on_row_activated)
        
        scrolled = Gtk.ScrolledWindow()
        scrolled.set_child(self.list_box)
        scrolled.set_margin_top(24); scrolled.set_margin_bottom(24)
        scrolled.set_margin_start(24); scrolled.set_margin_end(24)
        
        page_list.set_child(scrolled)
        split.set_content(page_list)

    def _refresh(self, *args):
        self.task_service.run_in_thread(self._load_tasks, self.task_service.get_all_tasks)

    def _load_tasks(self, tasks):
        mode = self.profile_service.get_current_mode()
        self.tasks_cache = [t for t in tasks if t.mode == mode]
        GLib.idle_add(self._update_calendar_marks)
        GLib.idle_add(self._update_list)

    def _update_calendar_marks(self):
        self.calendar.clear_marks()
        current_dt = self.calendar.get_date()
        for t in self.tasks_cache:
            if t.due_date:
                d = t.due_date
                if d.year == current_dt.get_year() and d.month == current_dt.get_month():
                    self.calendar.mark_day(d.day)

    def _on_date_changed(self, cal):
        dt = cal.get_date()
        self.selected_date = date(dt.get_year(), dt.get_month(), dt.get_day_of_month())
        self._update_list()

    def _update_list(self):
        child = self.list_box.get_first_child()
        while child:
            next_child = child.get_next_sibling()
            self.list_box.remove(child)
            child = next_child
            
        daily_tasks = [t for t in self.tasks_cache if t.due_date == self.selected_date]
        daily_tasks.sort(key=lambda t: t.priority.value, reverse=True)
        
        if not daily_tasks:
            empty = EmptyStateWidget(
                "x-office-calendar-symbolic",
                "No tasks this day",
                "Enjoy your free time!"
            )
            # ListBox expects a row, so wrap it
            row = Gtk.ListBoxRow(); row.set_activatable(False)
            row.set_child(empty)
            self.list_box.append(row)
        else:
            for t in daily_tasks:
                row = Gtk.ListBoxRow()
                row.task = t
                w = TaskRowWidget(t)
                row.set_child(w)
                self.list_box.append(row)

    def _on_row_activated(self, box, row):
        if hasattr(row, 'task'):
            win = self.get_root()
            dialog = TaskEditorDialog(win, self.task_service, self.board_service, self.profile_service, row.task)
            dialog.present()
