# rush/ui/views/dashboard_view.py
import gi
import webbrowser
from datetime import datetime
gi.require_version('Gtk', '4.0')
gi.require_version('Adw', '1')
from gi.repository import Gtk, Adw, Gdk, GLib

from rush.utils.event_bus import EventBus
from rush.ui.widgets.task_row import TaskRowWidget 
from rush.ui.dialogs.task_editor import TaskEditorDialog
from rush.data.models import Status

class DashboardView(Adw.Bin):
    __gtype_name__ = 'DashboardView'

    def __init__(self, task_service, profile_service, flow_service, board_service):
        super().__init__()
        self.task_service = task_service
        self.profile_service = profile_service
        self.flow_service = flow_service
        self.board_service = board_service
        self.bus = EventBus.get_default()
        
        self._setup_ui()
        
        self.bus.connect('task-updated', self._refresh)
        self.bus.connect('task-created', self._refresh)
        self.bus.connect('task-deleted', self._refresh)
        self.bus.connect('profile-updated', self._refresh)
        self._refresh()

    def _setup_ui(self):
        scrolled = Gtk.ScrolledWindow()
        self.set_child(scrolled)
        
        clamp = Adw.Clamp(maximum_size=900)
        scrolled.set_child(clamp)
        
        box = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=32)
        box.set_margin_top(48); box.set_margin_bottom(48)
        box.set_margin_start(24); box.set_margin_end(24)
        clamp.set_child(box)

        # --- Header Section ---
        header_box = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=12)
        
        vbox_labels = Gtk.Box(orientation=Gtk.Orientation.VERTICAL)
        self.lbl_greeting = Gtk.Label(css_classes=["dashboard-greeting"])
        self.lbl_greeting.set_halign(Gtk.Align.START)
        vbox_labels.append(self.lbl_greeting)
        
        self.lbl_sub = Gtk.Label(css_classes=["title-2", "dim-label"])
        self.lbl_sub.set_halign(Gtk.Align.START)
        vbox_labels.append(self.lbl_sub)
        
        header_box.append(vbox_labels)
        
        spacer = Gtk.Label(hexpand=True)
        header_box.append(spacer)
        
        btn_donate = Gtk.Button(label="Donate", icon_name="emblem-favorite-symbolic")
        btn_donate.add_css_class("pill")
        btn_donate.add_css_class("suggested-action")
        btn_donate.set_valign(Gtk.Align.CENTER)
        btn_donate.connect("clicked", lambda x: webbrowser.open("https://rush.manzoorahmed.com/donate.html")) 
        header_box.append(btn_donate)
        
        box.append(header_box)

        # --- Weekly Goal ---
        goal_box = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=12)
        goal_header = Gtk.Label(label="Weekly Goal", css_classes=["heading"])
        goal_header.set_halign(Gtk.Align.START)
        goal_box.append(goal_header)
        
        self.goal_progress = Gtk.ProgressBar()
        self.goal_progress.set_show_text(True)
        goal_box.append(self.goal_progress)
        
        box.append(goal_box)

        # --- Stats Grid ---
        grid_stats = Gtk.Box(spacing=20)
        grid_stats.set_homogeneous(True)
        
        # Color classes map to border-top colors in CSS
        self.card_flow = self._create_stat_card("Focus Today", "0m", "media-playback-start-symbolic", "stat-purple")
        grid_stats.append(self.card_flow)
        
        self.card_due = self._create_stat_card("Due Soon", "0", "alarm-symbolic", "stat-warning")
        grid_stats.append(self.card_due)
        
        self.card_done = self._create_stat_card("Completed", "0", "object-select-symbolic", "stat-success")
        grid_stats.append(self.card_done)
        
        box.append(grid_stats)

        # --- Up Next List ---
        lbl_next = Gtk.Label(label="Up Next", css_classes=["title-2"])
        lbl_next.set_halign(Gtk.Align.START)
        lbl_next.set_margin_top(16)
        box.append(lbl_next)
        
        self.list_box_next = Gtk.ListBox()
        self.list_box_next.add_css_class("boxed-list")
        self.list_box_next.set_selection_mode(Gtk.SelectionMode.NONE)
        self.list_box_next.connect("row-activated", self._on_task_activated)
        box.append(self.list_box_next)

    def _create_stat_card(self, title, value, icon, style_class):
        # We use a custom box to allow CSS styling on the container
        card = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=12)
        card.add_css_class("stat-card")
        card.add_css_class(style_class)
        
        # Header (Icon + Title)
        header = Gtk.Box(spacing=8)
        icon_w = Gtk.Image(icon_name=icon, pixel_size=16)
        icon_w.add_css_class("dim-label")
        
        lbl_title = Gtk.Label(label=title.upper(), css_classes=["caption-heading", "dim-label"])
        
        header.append(icon_w)
        header.append(lbl_title)
        card.append(header)
        
        # Value
        lbl_val = Gtk.Label(label=value, css_classes=["stat-value"])
        lbl_val.set_halign(Gtk.Align.START)
        card.append(lbl_val)
        
        return card

    def _refresh(self, *args):
        def fetch_data():
            tasks = self.task_service.get_all_tasks()
            focus_mins = self.flow_service.get_today_stats()
            profile = self.profile_service.get_user_profile()
            return tasks, focus_mins, profile

        self.task_service.run_in_thread(self._on_data_loaded, fetch_data)

    def _on_data_loaded(self, data):
        if not data: return
        tasks, focus_minutes, profile = data
        self._update_ui_safely(tasks, focus_minutes, profile)

    def _update_ui_safely(self, tasks, focus_minutes, profile):
        name = profile.get('name', 'User').split(' ')[0]
        hour = datetime.now().hour
        mode = profile.get('current_mode', 'Personal')
        
        if 5 <= hour < 12: greet = "Good morning"
        elif 12 <= hour < 18: greet = "Good afternoon"
        else: greet = "Good evening"
        
        self.lbl_greeting.set_label(f"{greet}, {name}.")
        self.lbl_sub.set_label(f"Current Mode: {mode}")

        tasks = [t for t in tasks if t.mode == mode]
        today_str = datetime.now().strftime("%Y-%m-%d")
        
        done_today = sum(1 for t in tasks if t.status == Status.DONE and t.completed_at and t.completed_at.strftime("%Y-%m-%d") == today_str)
        due_soon = sum(1 for t in tasks if t.status != Status.DONE and t.due_date and t.due_date.isoformat() == today_str)
        
        goal = int(profile.get('weekly_goal', 20))
        fraction = min(1.0, done_today / goal) if goal > 0 else 0
        self.goal_progress.set_fraction(fraction)
        self.goal_progress.set_text(f"{done_today} / {goal} Tasks")

        self._update_card_value(self.card_flow, f"{focus_minutes}m")
        self._update_card_value(self.card_done, str(done_today))
        self._update_card_value(self.card_due, str(due_soon))
        
        # Up Next List logic (preserved from previous fix)
        todos = [t for t in tasks if t.status != Status.DONE]
        todos.sort(key=lambda t: (-t.priority.value, t.due_date or datetime.max.date()))
        
        root = self.get_root()
        if root:
            focus_widget = root.get_focus()
            if focus_widget and focus_widget.is_ancestor(self.list_box_next):
                root.set_focus(None)
        
        self.list_box_next.unselect_all()
        child = self.list_box_next.get_first_child()
        while child:
            next_child = child.get_next_sibling()
            self.list_box_next.remove(child)
            child = next_child
        
        if not todos:
            box = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=12)
            box.set_margin_top(24); box.set_margin_bottom(24)
            img = Gtk.Image(icon_name="object-select-symbolic", pixel_size=48)
            img.add_css_class("dim-label")
            lbl = Gtk.Label(label=f"All caught up in {mode}!", css_classes=["title-2", "dim-label"])
            box.append(img); box.append(lbl)
            row = Gtk.ListBoxRow(); row.set_activatable(False); row.set_child(box)
            self.list_box_next.append(row)
        else:
            for t in todos[:5]:
                row = Gtk.ListBoxRow()
                row.task = t
                w = TaskRowWidget(t)
                row.set_child(w)
                self.list_box_next.append(row)

    def _update_card_value(self, card, value):
        # Card structure: [Header, Value]
        lbl = card.get_last_child()
        lbl.set_label(value)

    def _on_task_activated(self, listbox, row):
        if hasattr(row, 'task'):
            win = self.get_root()
            dialog = TaskEditorDialog(win, self.task_service, self.board_service, self.profile_service, row.task)
            dialog.present()
