# rush/ui/views/flow_view.py
import gi
gi.require_version('Gtk', '4.0')
gi.require_version('Adw', '1')
from gi.repository import Gtk, Adw, GLib

from rush.utils.event_bus import EventBus
from rush.data.models import Status

class FlowView(Adw.Bin):
    __gtype_name__ = 'FlowView'

    def __init__(self, flow_service, task_service):
        super().__init__()
        self.flow_service = flow_service
        self.task_service = task_service
        self.bus = EventBus.get_default()
        
        self.timer_active = False
        self.elapsed_seconds = 25 * 60 
        self.initial_duration = 25 * 60
        self.timer_id = None
        self.selected_task_id = None
        self.current_session_id = None 
        
        self._setup_ui()
        
        self.bus.connect('task-updated', self._refresh_tasks)
        self.bus.connect('task-created', self._refresh_tasks)
        self.bus.connect('task-deleted', self._refresh_tasks)
        self._refresh_tasks()

    def _setup_ui(self):
        box = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=32)
        box.set_halign(Gtk.Align.CENTER)
        box.set_valign(Gtk.Align.CENTER)
        box.set_margin_top(64); box.set_margin_bottom(64)
        
        self.lbl_timer = Gtk.Label(label="25:00", css_classes=["timer-display", "numeric"])
        box.append(self.lbl_timer)
        
        self.combo_task = Gtk.ComboBoxText()
        self.combo_task.set_size_request(300, -1)
        self.combo_task.connect("changed", self._on_task_selected)
        box.append(self.combo_task)
        
        btn_box = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=24)
        btn_box.set_halign(Gtk.Align.CENTER)
        
        self.btn_toggle = Gtk.Button()
        self.btn_toggle.add_css_class("circular")
        self.btn_toggle.add_css_class("suggested-action")
        self.btn_toggle.set_size_request(80, 80)
        
        icon_play = Gtk.Image(icon_name="media-playback-start-symbolic")
        icon_play.set_pixel_size(32)
        self.btn_toggle.set_child(icon_play)
        self.btn_toggle.connect("clicked", self._on_toggle_timer)
        btn_box.append(self.btn_toggle)
        
        btn_stop = Gtk.Button(icon_name="media-playback-stop-symbolic")
        btn_stop.add_css_class("circular")
        btn_stop.set_size_request(60, 60)
        btn_stop.connect("clicked", self._on_stop_timer)
        btn_box.append(btn_stop)
        
        box.append(btn_box)
        
        preset_box = Gtk.Box(spacing=12)
        preset_box.set_halign(Gtk.Align.CENTER)
        for mins in [15, 25, 45, 60]:
            btn = Gtk.Button(label=f"{mins}m")
            btn.add_css_class("flat")
            btn.connect("clicked", self._set_preset, mins)
            preset_box.append(btn)
        box.append(preset_box)
        
        self.set_child(box)

    def _refresh_tasks(self, *args):
        self.task_service.run_in_thread(self._populate_tasks, self.task_service.get_all_tasks)

    def _populate_tasks(self, tasks):
        # Filter for TODO/IN_PROGRESS
        active_tasks = [t for t in tasks if t.status != Status.DONE]
        
        def update_ui():
            current = self.combo_task.get_active_id()
            self.combo_task.remove_all()
            self.combo_task.append("none", "Select a task to focus on...")
            
            for t in active_tasks:
                self.combo_task.append(t.id, t.title)
            
            # Restore selection if it still exists
            if current and any(t.id == current for t in active_tasks):
                self.combo_task.set_active_id(current)
            else:
                self.combo_task.set_active_id("none")
                
        GLib.idle_add(update_ui)

    def _on_task_selected(self, combo):
        tid = combo.get_active_id()
        self.selected_task_id = tid if tid and tid != "none" else None

    def _set_preset(self, btn, mins):
        if not self.timer_active:
            self.initial_duration = mins * 60
            self.elapsed_seconds = self.initial_duration
            self._update_label()

    def _update_label(self):
        mins, secs = divmod(self.elapsed_seconds, 60)
        self.lbl_timer.set_label(f"{mins:02d}:{secs:02d}")

    def _on_toggle_timer(self, btn):
        if self.timer_active:
            # PAUSE
            self.timer_active = False
            self._update_toggle_button("media-playback-start-symbolic", True)
            if self.timer_id:
                GLib.source_remove(self.timer_id)
                self.timer_id = None
        else:
            # START
            self.timer_active = True
            self._update_toggle_button("media-playback-pause-symbolic", False)
            self.timer_id = GLib.timeout_add(1000, self._tick)
            
            # Start DB Session if not already running
            if self.selected_task_id and not self.current_session_id:
                self.flow_service.run_in_thread(
                    self._on_session_started,
                    self.flow_service.start_session,
                    self.selected_task_id, 
                    int(self.elapsed_seconds/60)
                )

    def _on_session_started(self, session_id):
        self.current_session_id = session_id

    def _update_toggle_button(self, icon_name, is_start):
        icon = Gtk.Image(icon_name=icon_name)
        icon.set_pixel_size(32)
        self.btn_toggle.set_child(icon)
        
        if is_start:
            self.btn_toggle.remove_css_class("destructive-action")
            self.btn_toggle.add_css_class("suggested-action")
        else:
            self.btn_toggle.remove_css_class("suggested-action")
            self.btn_toggle.add_css_class("destructive-action")

    def _tick(self):
        if self.elapsed_seconds > 0:
            self.elapsed_seconds -= 1
            self._update_label()
            return True
        else:
            self._on_stop_timer(None)
            return False

    def _on_stop_timer(self, btn):
        self.timer_active = False
        if self.timer_id:
            GLib.source_remove(self.timer_id)
            self.timer_id = None
            
        self._update_toggle_button("media-playback-start-symbolic", True)
        
        if self.current_session_id:
            actual_seconds = self.initial_duration - self.elapsed_seconds
            if actual_seconds > 0:
                self.flow_service.run_in_thread(
                    None,
                    self.flow_service.complete_session,
                    self.current_session_id,
                    actual_seconds
                )
            self.current_session_id = None

        self.initial_duration = 25 * 60
        self.elapsed_seconds = self.initial_duration
        self._update_label()
