# rush/ui/views/graph_view.py
import gi
import math
gi.require_version('Gtk', '4.0')
gi.require_version('Adw', '1')
from gi.repository import Gtk, Adw, Gdk, GLib

from rush.utils.event_bus import EventBus
from rush.ui.widgets.graph_node import GraphNodeWidget
from rush.ui.dialogs.task_editor import TaskEditorDialog

class GraphView(Adw.Bin):
    __gtype_name__ = 'GraphView'

    def __init__(self, task_service, board_service, profile_service):
        super().__init__()
        self.task_service = task_service
        self.board_service = board_service
        self.profile_service = profile_service
        self.bus = EventBus.get_default()
        
        self.selected_board_id = "all"
        self.boards_cache = []
        self.tasks = {}
        self.edges = []
        self.nodes_map = {} 
        
        # Scroll State
        self.scroll_step = 0
        self.scroll_source_id = None
        
        self._setup_ui()
        
        self.bus.connect('task-updated', self._refresh)
        self.bus.connect('task-created', self._refresh)
        self.bus.connect('task-deleted', self._refresh)
        self.bus.connect('board-changed', self._refresh)
        self._refresh()

    def _setup_ui(self):
        wrapper = Gtk.Box(orientation=Gtk.Orientation.VERTICAL)
        self.set_child(wrapper)
        
        toolbar = Gtk.Box(spacing=12)
        toolbar.set_margin_top(12); toolbar.set_margin_bottom(12)
        toolbar.set_margin_start(12); toolbar.set_margin_end(12)
        
        lbl = Gtk.Label(label="Graph View", css_classes=["heading"])
        toolbar.append(lbl)
        
        self.combo_board = Gtk.ComboBoxText()
        self.combo_board.set_size_request(200, -1)
        self.combo_board.connect("changed", self._on_board_filter_changed)
        toolbar.append(self.combo_board)
        
        help_btn = Gtk.MenuButton(icon_name="help-faq-symbolic")
        help_btn.set_tooltip_text("Drag from a node's Right Dot (Output) to another node to link them.")
        help_btn.add_css_class("flat")
        toolbar.append(help_btn)
        
        wrapper.append(toolbar)
        
        self.scrolled = Gtk.ScrolledWindow()
        self.scrolled.set_vexpand(True)
        wrapper.append(self.scrolled)
        
        # --- AUTO SCROLL DROP TARGET ---
        # This target catches drags anywhere in the viewport to trigger scrolling
        scroll_target = Gtk.DropTarget.new(type=str, actions=Gdk.DragAction.COPY)
        scroll_target.connect("enter", self._on_drag_enter_scroll)
        scroll_target.connect("motion", self._on_drag_motion_scroll)
        scroll_target.connect("leave", self._on_drag_leave_scroll)
        self.scrolled.add_controller(scroll_target)
        
        self.overlay = Gtk.Overlay()
        self.scrolled.set_child(self.overlay)
        
        self.drawing_area = Gtk.DrawingArea()
        self.drawing_area.set_content_width(3000)
        self.drawing_area.set_content_height(3000)
        self.drawing_area.set_draw_func(self._draw_lines)
        self.overlay.set_child(self.drawing_area)
        
        self.fixed = Gtk.Fixed()
        self.overlay.add_overlay(self.fixed)

    def _refresh(self, *args):
        self.task_service.run_in_thread(self._layout_graph, self._fetch_data)

    def _fetch_data(self):
        tasks_list = self.task_service.get_all_tasks()
        edges = self.task_service.get_all_dependencies()
        boards = self.board_service.get_all_boards()
        return tasks_list, edges, boards

    def _layout_graph(self, data):
        tasks_list, raw_edges, boards = data
        self.boards_cache = boards
        
        if self.profile_service:
            mode = self.profile_service.get_current_mode()
            tasks_list = [t for t in tasks_list if t.mode == mode]
        
        self.tasks = {t.id: t for t in tasks_list}
        self.edges = [e for e in raw_edges if e[0] in self.tasks and e[1] in self.tasks]

        layout_data = [] 
        labels_to_create = []
        
        X_SPACING = 260
        Y_SPACING = 110
        current_y_offset = 50
        max_width = 0
        
        boards_to_show = []
        if self.selected_board_id == "all":
            boards_to_show = boards + [{'id': 'uncategorized', 'name': 'Uncategorized'}]
        else:
            target = next((b for b in boards if b.id == self.selected_board_id), None)
            if target: boards_to_show = [target]
            elif self.selected_board_id == 'uncategorized':
                boards_to_show = [{'id': 'uncategorized', 'name': 'Uncategorized'}]
        
        board_tasks_map = {b['id'] if isinstance(b, dict) else b.id: [] for b in boards_to_show}
        for t in tasks_list:
            bid = t.board_id if t.board_id else 'uncategorized'
            if bid in board_tasks_map:
                board_tasks_map[bid].append(t)

        for board in boards_to_show:
            bid = board['id'] if isinstance(board, dict) else board.id
            b_name = board['name'] if isinstance(board, dict) else board.name
            b_tasks = board_tasks_map.get(bid, [])
            
            if not b_tasks and self.selected_board_id == "all": continue

            labels_to_create.append((b_name, current_y_offset))
            current_y_offset += 50
            
            local_tasks = {t.id: t for t in b_tasks}
            local_ids = set(local_tasks.keys())
            local_edges = [e for e in self.edges if e[0] in local_ids and e[1] in local_ids]
            
            depths = {tid: 0 for tid in local_ids}
            for _ in range(len(local_ids)):
                changed = False
                for parent, child in local_edges:
                    if depths[child] < depths[parent] + 1:
                        depths[child] = depths[parent] + 1
                        changed = True
                if not changed: break
            
            layers = {}
            for tid, d in depths.items():
                if d not in layers: layers[d] = []
                layers[d].append(tid)

            board_h = 0
            for layer_idx, task_ids in layers.items():
                task_ids.sort(key=lambda tid: local_tasks[tid].status.value)
                for row_idx, t_id in enumerate(task_ids):
                    task = local_tasks[t_id]
                    x = 50 + (layer_idx * X_SPACING)
                    y = current_y_offset + (row_idx * Y_SPACING)
                    
                    layout_data.append((task, x, y))
                    max_width = max(max_width, x + 250)
                    board_h = max(board_h, row_idx * Y_SPACING)
            
            current_y_offset += board_h + 100

        GLib.idle_add(self._apply_layout, layout_data, labels_to_create, max_width, current_y_offset)

    def _apply_layout(self, layout_data, labels_data, w, h):
        if self.combo_board.get_model() is None or self.combo_board.get_model().iter_n_children(None) == 0:
            self.combo_board.remove_all()
            self.combo_board.append("all", "All Boards")
            for b in self.boards_cache:
                self.combo_board.append(b.id, b.name)
            self.combo_board.append("uncategorized", "Uncategorized")
            self.combo_board.set_active_id(self.selected_board_id)

        child = self.fixed.get_first_child()
        while child:
            next_child = child.get_next_sibling()
            self.fixed.remove(child)
            child = next_child
        
        self.nodes_map = {}
        
        for text, y in labels_data:
            lbl = Gtk.Label(label=text)
            lbl.add_css_class("title-2")
            lbl.add_css_class("dim-label")
            self.fixed.put(lbl, 20, y)
        
        for task, x, y in layout_data:
            node = GraphNodeWidget(task, self._on_node_click, self._on_link_drop, self._on_status_toggle)
            node.node_x = x
            node.node_y = y
            self.fixed.put(node, x, y)
            self.nodes_map[task.id] = node

        self.drawing_area.set_content_width(max(w, 1200))
        self.drawing_area.set_content_height(max(h, 800))
        self.drawing_area.queue_draw()

    # --- AUTO SCROLL LOGIC ---
    def _on_drag_enter_scroll(self, target, x, y):
        self.scroll_step = 0
        return Gdk.DragAction.COPY

    def _on_drag_motion_scroll(self, target, x, y):
        # Detect proximity to edges
        height = self.scrolled.get_height()
        width = self.scrolled.get_width()
        margin = 60 # pixels from edge to trigger scroll
        step = 0
        
        # Vertical
        self.scroll_dy = 0
        if y < margin: self.scroll_dy = -10
        elif y > height - margin: self.scroll_dy = 10
        
        # Horizontal
        self.scroll_dx = 0
        if x < margin: self.scroll_dx = -10
        elif x > width - margin: self.scroll_dx = 10
        
        if (self.scroll_dx != 0 or self.scroll_dy != 0) and self.scroll_source_id is None:
            self.scroll_source_id = GLib.timeout_add(50, self._do_auto_scroll)
        
        if self.scroll_dx == 0 and self.scroll_dy == 0 and self.scroll_source_id:
            GLib.source_remove(self.scroll_source_id)
            self.scroll_source_id = None

        return Gdk.DragAction.COPY

    def _on_drag_leave_scroll(self, target):
        if self.scroll_source_id:
            GLib.source_remove(self.scroll_source_id)
            self.scroll_source_id = None

    def _do_auto_scroll(self):
        hadj = self.scrolled.get_hadjustment()
        vadj = self.scrolled.get_vadjustment()
        
        if self.scroll_dx != 0:
            hadj.set_value(hadj.get_value() + self.scroll_dx)
        if self.scroll_dy != 0:
            vadj.set_value(vadj.get_value() + self.scroll_dy)
            
        return True # Continue loop

    # --- ACTIONS ---

    def _on_board_filter_changed(self, combo):
        sid = combo.get_active_id()
        if sid:
            self.selected_board_id = sid
            self._refresh()

    def _on_link_drop(self, parent_id, child_id):
        self.task_service.add_dependency(parent_id, child_id)
        self.bus.emit('task-updated')

    def _on_status_toggle(self, task, is_complete):
        from rush.data.models import Status
        new_status = Status.DONE if is_complete else Status.TODO
        task.status = new_status
        
        # Update timestamp if completing
        if is_complete:
            from datetime import datetime
            task.completed_at = datetime.now()
        else:
            task.completed_at = None
            
        self.task_service.run_in_thread(
            lambda r: self.bus.emit('task-updated'), 
            self.task_service.update_task, 
            task
        )

    def _draw_lines(self, area, cr, w, h, data=None):
        if not hasattr(self, 'nodes_map') or not self.nodes_map:
            return

        cr.set_source_rgba(0.5, 0.5, 0.5, 0.6) 
        cr.set_line_width(2)
        
        NODE_W = 204
        NODE_H = 80 # Adjusted for new height with checkbox
        
        for parent_id, child_id in self.edges:
            if parent_id not in self.nodes_map or child_id not in self.nodes_map:
                continue
                
            p_node = self.nodes_map[parent_id]
            c_node = self.nodes_map[child_id]
            
            x1 = p_node.node_x + NODE_W - 12
            y1 = p_node.node_y + (NODE_H / 2)
            
            x2 = c_node.node_x + 12
            y2 = c_node.node_y + (NODE_H / 2)
            
            cp1_x = x1 + 50
            cp1_y = y1
            cp2_x = x2 - 50
            cp2_y = y2
            
            cr.move_to(x1, y1)
            cr.curve_to(cp1_x, cp1_y, cp2_x, cp2_y, x2, y2)
            cr.stroke()
            
            cr.save()
            cr.translate(x2, y2)
            cr.move_to(0, 0)
            cr.line_to(-8, -4)
            cr.line_to(-8, 4)
            cr.close_path()
            cr.fill()
            cr.restore()

    def _on_node_click(self, task):
        win = self.get_root()
        dialog = TaskEditorDialog(win, self.task_service, self.board_service, self.profile_service, task)
        dialog.present()
