# rush/ui/views/kanban_view.py
import gi
gi.require_version('Gtk', '4.0')
gi.require_version('Adw', '1')
from gi.repository import Gtk, Adw, Gio, GLib

from rush.ui.widgets.kanban_column import KanbanColumnWidget
from rush.utils.event_bus import EventBus

class KanbanView(Adw.Bin):
    __gtype_name__ = 'KanbanView'

    def __init__(self, task_service, board_service, profile_service):
        super().__init__()
        self.task_service = task_service
        self.board_service = board_service
        self.profile_service = profile_service
        self.bus = EventBus.get_default()
        
        self.board_widgets = {} 
        self.current_board_id = None
        self.boards_cache = []
        
        self._setup_ui()
        
        self.bus.connect('board-changed', self._refresh_boards)
        self.bus.connect('task-updated', self._refresh_tasks)
        self.bus.connect('task-created', self._refresh_tasks)
        self.bus.connect('task-deleted', self._refresh_tasks)
        
        self._refresh_boards()

    def _setup_ui(self):
        vbox = Gtk.Box(orientation=Gtk.Orientation.VERTICAL)
        self.set_child(vbox)
        
        toolbar = Gtk.Box(spacing=12)
        toolbar.set_margin_start(12); toolbar.set_margin_end(12)
        toolbar.set_margin_top(8); toolbar.set_margin_bottom(8)
        
        self.combo_board = Gtk.ComboBoxText()
        self.combo_board.set_size_request(200, -1)
        self.combo_board.connect("changed", self._on_board_changed)
        toolbar.append(self.combo_board)
        
        btn_add_col = Gtk.Button(icon_name="list-add-symbolic", tooltip_text="Add Column")
        btn_add_col.add_css_class("flat")
        btn_add_col.connect("clicked", self._on_add_column)
        toolbar.append(btn_add_col)
        
        # Spacer
        toolbar.append(Gtk.Box(hexpand=True))
        
        btn_del_board = Gtk.Button(icon_name="user-trash-symbolic", tooltip_text="Delete Board")
        btn_del_board.add_css_class("flat")
        btn_del_board.add_css_class("destructive-action")
        btn_del_board.connect("clicked", self._on_delete_board)
        toolbar.append(btn_del_board)

        btn_new_board = Gtk.Button(label="New Board")
        btn_new_board.add_css_class("flat")
        btn_new_board.connect("clicked", self._on_new_board)
        toolbar.append(btn_new_board)
        
        vbox.append(toolbar)
        
        self.scrolled = Gtk.ScrolledWindow()
        self.scrolled.set_vexpand(True)
        vbox.append(self.scrolled)
        
        self.board_box = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=16)
        self.board_box.set_margin_top(16); self.board_box.set_margin_bottom(16)
        self.board_box.set_margin_start(16); self.board_box.set_margin_end(16)
        self.scrolled.set_child(self.board_box)

    def _refresh_boards(self, *args):
        self.task_service.run_in_thread(self._load_boards, self.board_service.get_all_boards)

    def _load_boards(self, boards):
        self.boards_cache = boards
        GLib.idle_add(self._update_combo_safely, boards)

    def _update_combo_safely(self, boards):
        # Preserve selection if possible
        active = self.combo_board.get_active_id() or self.current_board_id
        
        self.combo_board.remove_all()
        for b in boards:
            self.combo_board.append(b.id, b.name)
        
        if boards:
            if active and any(b.id == active for b in boards):
                self.combo_board.set_active_id(active)
            else:
                self.combo_board.set_active_id(boards[0].id)
        else:
            self.current_board_id = None
            self._build_columns()

    def _on_board_changed(self, combo):
        bid = combo.get_active_id()
        if bid:
            self.current_board_id = bid
            self._build_columns()

    def _build_columns(self):
        # Clear existing
        child = self.board_box.get_first_child()
        while child:
            next_child = child.get_next_sibling()
            self.board_box.remove(child)
            child = next_child
        
        self.board_widgets = {}
        
        board = next((b for b in self.boards_cache if b.id == self.current_board_id), None)
        if not board: return
        
        for col in board.columns:
            cw = KanbanColumnWidget(
                col.id, col.name, col.color, 
                self.task_service, self.board_service, self.get_root(),
                self.profile_service
            )
            cw.board_id = board.id
            self.board_box.append(cw)
            self.board_widgets[col.id] = cw
            
        self._refresh_tasks()

    def _refresh_tasks(self, *args):
        if not self.current_board_id: return
        self.task_service.run_in_thread(self._distribute_tasks, self.task_service.get_kanban_board_tasks, self.current_board_id)

    def _distribute_tasks(self, grouped_tasks):
        mode = self.profile_service.get_current_mode() if self.profile_service else "Personal"
        
        def update_ui():
            for col_id, widget in self.board_widgets.items():
                tasks = grouped_tasks.get(col_id, [])
                tasks = [t for t in tasks if t.mode == mode]
                widget.update_tasks(tasks)
        GLib.idle_add(update_ui)

    def _on_add_column(self, btn):
        if not self.current_board_id: return
        dialog = Adw.MessageDialog(heading="New Column", body="Enter column name")
        dialog.set_transient_for(self.get_root())
        
        entry = Gtk.Entry()
        dialog.set_extra_child(entry)
        dialog.add_response("cancel", "Cancel")
        dialog.add_response("add", "Add")
        dialog.set_response_appearance("add", Adw.ResponseAppearance.SUGGESTED)
        
        def on_resp(d, r):
            if r == "add" and entry.get_text():
                self.board_service.add_column(self.current_board_id, entry.get_text())
            d.close()
        dialog.connect("response", on_resp)
        dialog.present()

    def _on_new_board(self, btn):
        dialog = Adw.MessageDialog(heading="New Board", body="Enter board name")
        dialog.set_transient_for(self.get_root())
        
        entry = Gtk.Entry()
        dialog.set_extra_child(entry)
        dialog.add_response("cancel", "Cancel")
        dialog.add_response("create", "Create")
        dialog.set_response_appearance("create", Adw.ResponseAppearance.SUGGESTED)
        
        def on_resp(d, r):
            if r == "create" and entry.get_text():
                self.board_service.create_board(entry.get_text())
            d.close()
        dialog.connect("response", on_resp)
        dialog.present()

    def _on_delete_board(self, btn):
        if not self.current_board_id: return
        
        # Don't delete if it's the only board
        if len(self.boards_cache) <= 1:
            d = Adw.MessageDialog(heading="Cannot Delete", body="You must have at least one board.", transient_for=self.get_root())
            d.add_response("ok", "OK")
            d.present()
            return

        dialog = Adw.MessageDialog(
            heading="Delete Board?", 
            body="Tasks in this board will be moved to 'Uncategorized'. They will NOT be deleted.",
            transient_for=self.get_root()
        )
        dialog.add_response("cancel", "Cancel")
        dialog.add_response("delete", "Delete")
        dialog.set_response_appearance("delete", Adw.ResponseAppearance.DESTRUCTIVE)
        
        def on_resp(d, r):
            if r == "delete":
                self.board_service.delete_board(self.current_board_id)
            d.close()
        
        dialog.connect("response", on_resp)
        dialog.present()
