# rush/ui/views/roadmap_view.py
import gi
import math
from datetime import datetime, timedelta, date
gi.require_version('Gtk', '4.0')
gi.require_version('Adw', '1')
from gi.repository import Gtk, Adw, Gdk, GLib

from rush.utils.event_bus import EventBus
from rush.ui.dialogs.task_editor import TaskEditorDialog

class RoadmapView(Adw.Bin):
    __gtype_name__ = 'RoadmapView'

    def __init__(self, task_service, board_service, profile_service):
        super().__init__()
        self.task_service = task_service
        self.board_service = board_service
        self.profile_service = profile_service
        self.bus = EventBus.get_default()
        
        self.tasks = []
        # Start view from 3 days ago to show immediate context
        self.start_date = date.today() - timedelta(days=3)
        self.view_duration = 35 # Show 35 days horizon
        
        self.day_width = 80 
        self.row_height = 50 
        self.header_height = 40
        
        self._setup_ui()
        self.bus.connect('task-updated', self._refresh)
        self.bus.connect('task-created', self._refresh)
        self.bus.connect('task-deleted', self._refresh)
        self._refresh()

    def _setup_ui(self):
        scrolled = Gtk.ScrolledWindow()
        self.set_child(scrolled)
        
        self.drawing_area = Gtk.DrawingArea()
        # Initial size, will resize dynamically
        self.drawing_area.set_content_width(self.day_width * self.view_duration)
        self.drawing_area.set_content_height(800) 
        self.drawing_area.set_draw_func(self._draw)
        
        # Click controller
        click = Gtk.GestureClick()
        click.connect("pressed", self._on_click)
        self.drawing_area.add_controller(click)
        
        scrolled.set_child(self.drawing_area)

    def _refresh(self, *args):
        self.task_service.run_in_thread(self._load, self.task_service.get_all_tasks)

    def _load(self, tasks):
        mode = self.profile_service.get_current_mode()
        
        # Calculate View Window
        view_start = self.start_date
        view_end = self.start_date + timedelta(days=self.view_duration)
        
        visible_tasks = []
        for t in tasks:
            # 1. Filter by Mode and Due Date existence
            if t.mode != mode or not t.due_date:
                continue
            
            # 2. Filter by Time Window Intersection
            task_start = t.created_at.date()
            task_end = t.due_date
            
            # Simple overlap check: StartA <= EndB and EndA >= StartB
            if task_start <= view_end and task_end >= view_start:
                visible_tasks.append(t)

        # Sort by due date, then priority
        visible_tasks.sort(key=lambda t: (t.due_date, t.priority.value))
        self.tasks = visible_tasks
        
        # Resize Canvas on Main Thread
        GLib.idle_add(self._resize_and_draw)

    def _resize_and_draw(self):
        # Calculate required height based on task count
        total_height = max(800, self.header_height + (len(self.tasks) * self.row_height) + 50)
        total_width = max(1200, self.day_width * self.view_duration)
        
        self.drawing_area.set_content_height(total_height)
        self.drawing_area.set_content_width(total_width)
        self.drawing_area.queue_draw()

    def _draw(self, area, cr, w, h):
        # Background
        cr.set_source_rgb(1, 1, 1)
        cr.paint()

        # --- Draw Header (Dates) ---
        cr.set_source_rgb(0.98, 0.98, 0.98) # Header BG
        cr.rectangle(0, 0, w, self.header_height)
        cr.fill()
        
        # Draw Grid & Dates
        cr.set_line_width(1)
        
        # Palette
        colors = [
            (0.2, 0.6, 1.0), # Blue
            (0.6, 0.4, 0.9), # Purple
            (0.1, 0.8, 0.7), # Teal
            (1.0, 0.6, 0.2), # Orange
            (0.9, 0.3, 0.4)  # Red
        ]
        
        for i in range(self.view_duration):
            d = self.start_date + timedelta(days=i)
            x = i * self.day_width
            
            # Date Text
            is_today = d == date.today()
            if is_today:
                cr.set_source_rgb(0.2, 0.2, 0.8) # Highlight Today
                cr.select_font_face("Sans", 0, 1) # Bold
            else:
                cr.set_source_rgb(0.4, 0.4, 0.4)
                cr.select_font_face("Sans", 0, 0) # Normal
                
            cr.move_to(x + 10, 26)
            cr.set_font_size(13)
            cr.show_text(d.strftime("%d %b"))
            
            # Vertical Grid Line
            cr.set_source_rgba(0.0, 0.0, 0.0, 0.05)
            if is_today: cr.set_source_rgba(0.2, 0.2, 0.8, 0.1) # Highlight Today Line
            
            cr.move_to(x, self.header_height)
            cr.line_to(x, h)
            cr.stroke()

        # --- Draw Tasks ---
        y = self.header_height + 10
        
        for i, t in enumerate(self.tasks):
            # Determine rendering X and Width relative to view
            task_start_date = t.created_at.date()
            task_end_date = t.due_date
            
            # Clamp start to view start
            render_start = max(task_start_date, self.start_date)
            # Clamp end to view end (for width calc only)
            view_end = self.start_date + timedelta(days=self.view_duration)
            render_end = min(task_end_date, view_end)
            
            offset_days = (render_start - self.start_date).days
            duration_days = (render_end - render_start).days
            
            # Ensure minimum visibility
            if duration_days < 1: duration_days = 1
            
            x = offset_days * self.day_width
            width = duration_days * self.day_width
            bar_h = 32
            
            # Indicate if task continues to left or right
            continues_left = task_start_date < self.start_date
            continues_right = task_end_date > view_end
            
            # Pick color
            c = colors[i % len(colors)]
            
            # Dim done
            if t.status.name == 'DONE':
                cr.set_source_rgba(0.6, 0.6, 0.6, 0.4)
            else:
                cr.set_source_rgb(*c)
            
            # Draw Bar
            radius = 8
            width = max(width, radius * 2) # Prevent weird arc math
            
            cr.new_sub_path()
            # If continues left, flat edge, else rounded
            if continues_left:
                cr.move_to(x, y + bar_h)
                cr.line_to(x, y)
            else:
                cr.arc(x + radius, y + radius, radius, math.pi, 3*math.pi/2)
                
            # If continues right, flat edge
            if continues_right:
                cr.line_to(x + width, y)
                cr.line_to(x + width, y + bar_h)
            else:
                cr.arc(x + width - radius, y + radius, radius, -math.pi/2, 0)
                cr.arc(x + width - radius, y + bar_h - radius, radius, 0, math.pi/2)
                
            if not continues_left:
                cr.arc(x + radius, y + bar_h - radius, radius, math.pi/2, math.pi)
            else:
                cr.line_to(x, y + bar_h)
                
            cr.close_path()
            
            # Shadow
            cr.save()
            cr.translate(2, 2)
            cr.set_source_rgba(0, 0, 0, 0.1)
            cr.fill_preserve()
            cr.restore()
            
            # Fill
            if t.status.name == 'DONE':
                cr.set_source_rgba(0.6, 0.6, 0.6, 0.4)
            else:
                cr.set_source_rgb(*c)
            cr.fill()
            
            # Text - FIX: Move setup INSIDE the save/clip block
            cr.save()
            
            # 1. Define clip
            cr.rectangle(x, y, width - 5, bar_h)
            cr.clip()
            
            # 2. Setup text style
            cr.set_source_rgb(1, 1, 1)
            cr.select_font_face("Sans", 0, 1)
            
            # 3. Move to position (Must happen after clip!)
            cr.move_to(x + 10, y + 21)
            
            # 4. Draw
            cr.show_text(t.title)
            
            cr.restore()
            
            # Store click rect
            t._render_rect = (x, y, width, bar_h)
            
            y += self.row_height

    def _on_click(self, gesture, n, x, y):
        for t in self.tasks:
            if hasattr(t, '_render_rect'):
                rx, ry, rw, rh = t._render_rect
                if rx <= x <= rx + rw and ry <= y <= ry + rh:
                    win = self.get_root()
                    dialog = TaskEditorDialog(win, self.task_service, self.board_service, self.profile_service, t)
                    dialog.present()
                    break
