# rush/ui/views/stats_view.py
import gi
from datetime import datetime, timedelta
gi.require_version('Gtk', '4.0')
gi.require_version('Adw', '1')
from gi.repository import Gtk, Adw, Gdk, GLib

from rush.utils.event_bus import EventBus
from rush.ui.widgets.charts import BarChartWidget, PieChartWidget
from rush.services.report_service import ReportService

class StatsView(Adw.Bin):
    __gtype_name__ = 'StatsView'

    def __init__(self, task_service, board_service, flow_service):
        super().__init__()
        self.task_service = task_service
        self.report_service = ReportService(task_service)
        self.bus = EventBus.get_default()
        
        self._setup_ui()
        self.bus.connect('task-updated', self._refresh)
        self.bus.connect('profile-updated', self._refresh)
        self._refresh()

    def _setup_ui(self):
        scrolled = Gtk.ScrolledWindow()
        self.set_child(scrolled)
        
        clamp = Adw.Clamp(maximum_size=1000)
        scrolled.set_child(clamp)
        
        box = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=32)
        box.set_margin_top(32); box.set_margin_bottom(32)
        box.set_margin_start(24); box.set_margin_end(24)
        clamp.set_child(box)

        # 1. Metrics Grid
        self.grid = Gtk.Box(spacing=20, homogeneous=True)
        self.card_rate = self._create_metric_card("Completion Rate", "0%", "stat-success")
        self.card_consistency = self._create_metric_card("Consistency", "0", "stat-accent")
        self.card_avg = self._create_metric_card("Avg Session", "0m", "stat-purple")
        
        self.grid.append(self.card_rate)
        self.grid.append(self.card_consistency)
        self.grid.append(self.card_avg)
        box.append(self.grid)

        # 2. Charts Section
        charts_container = Gtk.Box(spacing=24, homogeneous=True)
        
        # Weekly Completion
        vbox_bar = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=16)
        vbox_bar.add_css_class("stat-card") # Card background for charts
        vbox_bar.append(Gtk.Label(label="Tasks (Last 7 Days)", css_classes=["heading"]))
        self.chart_bar = BarChartWidget()
        self.chart_bar.set_size_request(-1, 220)
        vbox_bar.append(self.chart_bar)
        charts_container.append(vbox_bar)

        # Focus Distribution
        vbox_pie = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=16)
        vbox_pie.add_css_class("stat-card")
        vbox_pie.append(Gtk.Label(label="Focus by Board", css_classes=["heading"]))
        self.chart_pie = PieChartWidget()
        self.chart_pie.set_size_request(-1, 220)
        vbox_pie.append(self.chart_pie)
        charts_container.append(vbox_pie)

        box.append(charts_container)

    def _create_metric_card(self, title, value, color_class):
        card = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=8)
        card.add_css_class("stat-card")
        card.add_css_class(color_class)
        
        lbl_title = Gtk.Label(label=title.upper(), css_classes=["caption-heading", "dim-label"])
        lbl_title.set_halign(Gtk.Align.START)
        
        lbl_val = Gtk.Label(label=value, css_classes=["stat-value"])
        lbl_val.set_halign(Gtk.Align.START)
        
        card.append(lbl_title)
        card.append(lbl_val)
        return card

    def _refresh(self, *args):
        self.task_service.run_in_thread(self._on_data_loaded, self._fetch_stats)

    def _fetch_stats(self):
        extended = self.report_service.get_extended_stats(days=7)
        trend = self.report_service.get_completion_trend(days=7)
        dist = self.report_service.get_time_distribution(days=7)
        return extended, trend, dist

    def _on_data_loaded(self, data):
        if not data: return
        extended, trend, dist = data
        
        self._update_label(self.card_rate, f"{extended.get('completion_rate', 0)}%")
        self._update_label(self.card_consistency, f"{extended.get('consistency_score', 0)}")
        self._update_label(self.card_avg, f"{extended.get('avg_session_len', 0)}m")
        
        if trend:
            bar_data = {d.split('-')[-1]: c for d, c in trend}
            self.chart_bar.set_data(bar_data)
        else:
            self.chart_bar.set_data({})

        self.chart_pie.set_data(dist if dist else {})

    def _update_label(self, card, text):
        lbl = card.get_last_child()
        lbl.set_label(text)
