# rush/ui/views/task_list_view.py
import gi
gi.require_version('Gtk', '4.0')
gi.require_version('Adw', '1')
from gi.repository import Gtk, Adw, Gio, GLib, GObject

from rush.ui.widgets.task_row import TaskRowWidget
from rush.ui.widgets.empty_state import EmptyStateWidget
from rush.ui.objects import TaskObject
from rush.ui.dialogs.task_editor import TaskEditorDialog
from rush.utils.event_bus import EventBus

class TaskListView(Adw.Bin):
    __gtype_name__ = 'TaskListView'

    def __init__(self, task_service, profile_service, board_service):
        super().__init__()
        self.task_service = task_service
        self.profile_service = profile_service
        self.board_service = board_service
        self.bus = EventBus.get_default()
        
        self.model = Gio.ListStore(item_type=TaskObject)
        self.filter_model = Gtk.FilterListModel(model=self.model)
        self.sort_model = Gtk.SortListModel(model=self.filter_model)
        self.selection_model = Gtk.SingleSelection(model=self.sort_model)
        
        self.current_filter = "active"
        
        self._setup_ui()
        
        self.bus.connect('task-updated', self._refresh)
        self.bus.connect('task-created', self._refresh)
        self.bus.connect('task-deleted', self._refresh)
        self.bus.connect('profile-updated', self._refresh)
        self._refresh()

    def _setup_ui(self):
        vbox = Gtk.Box(orientation=Gtk.Orientation.VERTICAL)
        self.set_child(vbox)
        
        # Toolbar
        toolbar = Gtk.Box(spacing=12)
        toolbar.set_margin_top(8); toolbar.set_margin_bottom(8)
        toolbar.set_margin_start(12); toolbar.set_margin_end(12)
        
        self.search_entry = Gtk.SearchEntry()
        self.search_entry.set_placeholder_text("Filter tasks...")
        self.search_entry.connect("search-changed", self._on_search_changed)
        toolbar.append(self.search_entry)
        
        filter_box = Gtk.Box(css_classes=["linked"])
        for label, val in [("Active", "active"), ("Done", "done"), ("All", "all")]:
            btn = Gtk.ToggleButton(label=label)
            btn.set_group(filter_box.get_first_child() if filter_box.get_first_child() else None)
            if val == "active": btn.set_active(True)
            btn.connect("toggled", self._on_filter_toggled, val)
            filter_box.append(btn)
        toolbar.append(filter_box)
        
        vbox.append(toolbar)
        
        # Stack to switch between List and Empty State
        self.stack = Gtk.Stack()
        vbox.append(self.stack)
        
        # 1. List View
        factory = Gtk.SignalListItemFactory()
        factory.connect("setup", self._on_setup_row)
        factory.connect("bind", self._on_bind_row)
        
        list_view = Gtk.ListView(model=self.selection_model, factory=factory)
        list_view.connect("activate", self._on_activate)
        
        scrolled = Gtk.ScrolledWindow()
        scrolled.set_vexpand(True)
        scrolled.set_child(list_view)
        self.stack.add_named(scrolled, "list")
        
        # 2. Empty State
        self.empty = EmptyStateWidget(
            "checkbox-checked-symbolic",
            "No tasks found",
            "Try adjusting filters or create a new task.",
            "New Task",
            self._on_quick_add
        )
        self.stack.add_named(self.empty, "empty")

    def _refresh(self, *args):
        self.task_service.run_in_thread(self._load_tasks, self.task_service.get_all_tasks)

    def _load_tasks(self, tasks):
        mode = self.profile_service.get_current_mode()
        filtered = [t for t in tasks if t.mode == mode]
        GLib.idle_add(self._update_model, filtered)

    def _update_model(self, tasks):
        self.model.remove_all()
        for t in tasks:
            self.model.append(TaskObject(t))
        self._apply_filter()

    def _on_filter_toggled(self, btn, val):
        if btn.get_active():
            self.current_filter = val
            self._apply_filter()

    def _on_search_changed(self, entry):
        self._apply_filter()

    def _apply_filter(self):
        query = self.search_entry.get_text().lower()
        
        def filter_func(item):
            task = item.task
            if self.current_filter == "active" and task.status.name == "DONE": return False
            if self.current_filter == "done" and task.status.name != "DONE": return False
            
            if query:
                in_title = query in task.title.lower()
                in_desc = query in task.description.lower()
                in_tags = any(query in tag.lower() for tag in task.tags)
                if not (in_title or in_desc or in_tags): return False
            return True

        self.filter_model.set_filter(Gtk.CustomFilter.new(filter_func))
        self.sort_model.set_sorter(Gtk.CustomSorter.new(self._sort_func))
        
        # Update Empty State visibility
        count = self.selection_model.get_n_items()
        self.stack.set_visible_child_name("empty" if count == 0 else "list")

    def _sort_func(self, a, b, *args):
        # FIX: Accepts *args to handle the user_data passed by GTK
        sa = 1 if a.task.status.name == 'DONE' else 0
        sb = 1 if b.task.status.name == 'DONE' else 0
        if sa != sb: return sa - sb
        
        pa = a.task.priority.value
        pb = b.task.priority.value
        if pa != pb: return pb - pa
        
        da = a.task.due_date.isoformat() if a.task.due_date else "9999"
        db = b.task.due_date.isoformat() if b.task.due_date else "9999"
        if da < db: return -1
        if da > db: return 1
        
        return 0

    def _on_setup_row(self, factory, item):
        row = TaskRowWidget(None) 
        item.set_child(row)

    def _on_bind_row(self, factory, item):
        widget = item.get_child()
        obj = item.get_item()
        widget.bind(obj.task)

    def _on_activate(self, listview, position):
        obj = self.sort_model.get_item(position)
        if obj:
            win = self.get_root()
            dialog = TaskEditorDialog(win, self.task_service, self.board_service, self.profile_service, obj.task)
            dialog.present()
    
    def _on_quick_add(self, btn):
        win = self.get_root()
        # Find the action or call dialog directly
        dialog = TaskEditorDialog(win, self.task_service, self.board_service, self.profile_service, None)
        dialog.present()
