# rush/ui/widgets/graph_node.py
import gi
gi.require_version('Gtk', '4.0')
gi.require_version('Adw', '1')
from gi.repository import Gtk, Gdk, Adw

from rush.data.models import Status

class GraphNodeWidget(Gtk.Box):
    def __init__(self, task, on_click_cb, on_drop_cb, on_status_toggle_cb):
        super().__init__(orientation=Gtk.Orientation.HORIZONTAL, spacing=0)
        self.task = task
        self.on_click_cb = on_click_cb
        self.on_drop_cb = on_drop_cb
        self.on_status_toggle_cb = on_status_toggle_cb
        
        self.node_x = 0
        self.node_y = 0
        
        self.add_css_class("graph-node-wrapper")
        
        # 1. INPUT PORT
        self.input_port = Gtk.Image(icon_name="media-record-symbolic", pixel_size=12)
        self.input_port.add_css_class("graph-port")
        self.input_port.add_css_class("port-input")
        self.input_port.set_valign(Gtk.Align.CENTER)
        self.input_port.set_tooltip_text("Input (Drop here)")
        self.append(self.input_port)
        
        # 2. CONTENT BOX
        self.content_box = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=4)
        self.content_box.add_css_class("graph-node-content")
        self.content_box.set_size_request(180, 70)
        
        # Header Row (Checkbox + Title)
        header = Gtk.Box(spacing=8)
        self.check = Gtk.CheckButton()
        self.check.set_active(task.status == Status.DONE)
        self.check.connect("toggled", self._on_toggled)
        self.check.set_tooltip_text("Mark as Complete")
        header.append(self.check)
        
        lbl_title = Gtk.Label(label=task.title, wrap=True, max_width_chars=18, xalign=0)
        lbl_title.add_css_class("graph-node-title")
        header.append(lbl_title)
        self.content_box.append(header)
        
        # Meta Row
        meta_box = Gtk.Box(spacing=6)
        lbl_status = Gtk.Label(label=task.status.name.title(), xalign=0)
        lbl_status.add_css_class("caption")
        lbl_status.add_css_class("dim-label")
        meta_box.append(lbl_status)
        
        if task.priority.value > 2:
            lbl_prio = Gtk.Label(label="!", css_classes=["prio-high"])
            meta_box.append(lbl_prio)
            
        self.content_box.append(meta_box)
        
        # Style if done
        if task.status == Status.DONE:
            self.content_box.add_css_class("graph-node-completed")
        
        self.append(self.content_box)
        
        # 3. OUTPUT PORT
        self.output_port = Gtk.Image(icon_name="media-record-symbolic", pixel_size=12)
        self.output_port.add_css_class("graph-port")
        self.output_port.add_css_class("port-output")
        self.output_port.set_valign(Gtk.Align.CENTER)
        self.output_port.set_tooltip_text("Output (Drag to link)")
        self.append(self.output_port)

        # --- CONTROLLERS ---
        click = Gtk.GestureClick()
        click.connect("pressed", self._on_press)
        self.content_box.add_controller(click)
        
        target = Gtk.DropTarget.new(type=str, actions=Gdk.DragAction.COPY)
        target.connect("drop", self._on_drop)
        self.add_controller(target)
        
        source = Gtk.DragSource()
        source.set_actions(Gdk.DragAction.COPY)
        source.connect("prepare", self._on_drag_prepare)
        self.output_port.add_controller(source)

    def _on_toggled(self, btn):
        is_done = btn.get_active()
        if is_done:
            self.content_box.add_css_class("graph-node-completed")
        else:
            self.content_box.remove_css_class("graph-node-completed")
            
        self.on_status_toggle_cb(self.task, is_done)

    def _on_press(self, gesture, n, x, y):
        self.on_click_cb(self.task)

    def _on_drop(self, target, value, x, y):
        source_id = value
        if source_id and source_id != self.task.id:
            self.on_drop_cb(source_id, self.task.id)
            return True
        return False

    def _on_drag_prepare(self, source, x, y):
        return Gdk.ContentProvider.new_for_value(self.task.id)
