# rush/ui/widgets/kanban_column.py
import gi
gi.require_version('Gtk', '4.0')
gi.require_version('Adw', '1')
from gi.repository import Gtk, Adw, Gdk, GLib

from rush.data.models import Status
from rush.ui.widgets.kanban_card import KanbanCardWidget
from rush.ui.dialogs.task_editor import TaskEditorDialog
from rush.utils.event_bus import EventBus

class KanbanColumnWidget(Gtk.Box):
    __gtype_name__ = 'KanbanColumnWidget'

    def __init__(self, column_id, title, color, task_service, board_service, root_window, profile_service):
        super().__init__(orientation=Gtk.Orientation.VERTICAL, spacing=12)
        self.column_id = column_id
        self.title = title
        self.color = color
        self.task_service = task_service
        self.board_service = board_service
        self.root_window = root_window
        self.profile_service = profile_service
        self.board_id = None 
        
        self.bus = EventBus.get_default()

        self.set_size_request(280, -1)
        self.add_css_class("kanban-column")
        
        self._setup_ui()
        self._setup_dnd()

    def _setup_ui(self):
        header = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=8)
        header.set_margin_top(8); header.set_margin_bottom(8)
        header.set_margin_start(8); header.set_margin_end(8)
        
        dot = Gtk.Image(icon_name="media-record-symbolic", pixel_size=12)
        dot.add_css_class(self.color if self.color else "accent")
        header.append(dot)
        
        lbl = Gtk.Label(label=self.title.upper(), css_classes=["heading", "dim-label"])
        header.append(lbl)
        
        self.lbl_count = Gtk.Label(label="0", css_classes=["caption", "dim-label"])
        self.lbl_count.set_hexpand(True)
        self.lbl_count.set_halign(Gtk.Align.END)
        header.append(self.lbl_count)
        
        btn_menu = Gtk.MenuButton(icon_name="view-more-symbolic")
        btn_menu.add_css_class("flat")
        btn_menu.set_popover(self._create_menu())
        header.append(btn_menu)
        
        self.append(header)
        
        scrolled = Gtk.ScrolledWindow()
        scrolled.set_vexpand(True)
        scrolled.set_policy(Gtk.PolicyType.NEVER, Gtk.PolicyType.AUTOMATIC)
        
        self.list_box = Gtk.ListBox()
        self.list_box.add_css_class("kanban-list")
        self.list_box.set_selection_mode(Gtk.SelectionMode.NONE)
        self.list_box.set_activate_on_single_click(False)
        
        scrolled.set_child(self.list_box)
        self.append(scrolled)

    def _create_menu(self):
        pop = Gtk.Popover()
        box = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=6)
        box.set_margin_top(6); box.set_margin_bottom(6)
        box.set_margin_start(6); box.set_margin_end(6)
        
        btn_ren = Gtk.Button(label="Rename", css_classes=["flat"])
        btn_ren.connect("clicked", self._on_rename)
        box.append(btn_ren)
        
        btn_del = Gtk.Button(label="Delete Column", css_classes=["flat", "destructive-action"])
        btn_del.connect("clicked", self._on_delete)
        box.append(btn_del)
        
        pop.set_child(box)
        return pop

    def _on_rename(self, btn):
        pop = btn.get_ancestor(Gtk.Popover)
        if pop: pop.popdown()
        
        dialog = Adw.MessageDialog(
            transient_for=self.root_window,
            heading="Rename Column",
            body=f"Rename '{self.title}'?"
        )
        entry = Gtk.Entry(text=self.title)
        dialog.set_extra_child(entry)
        dialog.add_response("cancel", "Cancel")
        dialog.add_response("rename", "Rename")
        dialog.set_response_appearance("rename", Adw.ResponseAppearance.SUGGESTED)
        
        def on_resp(d, r):
            if r == "rename" and self.board_service:
                self.board_service.update_column(self.column_id, entry.get_text())
            d.close()
            
        dialog.connect("response", on_resp)
        dialog.present()

    def _on_delete(self, btn):
        pop = btn.get_ancestor(Gtk.Popover)
        if pop: pop.popdown()

        if self.board_service:
            self.board_service.delete_column(self.column_id)

    def _setup_dnd(self):
        target = Gtk.DropTarget.new(type=str, actions=Gdk.DragAction.MOVE)
        target.connect("drop", self._on_drop)
        self.add_controller(target)

    def update_tasks(self, tasks):
        # --- FOCUS SAFETY FIX ---
        root = self.get_root()
        if root:
            focus_widget = root.get_focus()
            if focus_widget and focus_widget.is_ancestor(self.list_box):
                root.set_focus(None)

        # Clear list
        child = self.list_box.get_first_child()
        while child:
            next_child = child.get_next_sibling()
            self.list_box.remove(child)
            child = next_child

        self.lbl_count.set_label(str(len(tasks)))
        
        for task in tasks:
            blockers = self.task_service.get_blockers(task.id)
            is_blocked = any(b.status != Status.DONE for b in blockers)
            row = KanbanTaskRow(task, self.task_service, self.board_service, self.profile_service, self.root_window, is_blocked)
            self.list_box.append(row)

    def _on_drop(self, target, value, x, y):
        task_id = value
        if not task_id: return False
        
        def do_move():
            self.task_service.move_task_to_column(task_id, self.column_id)
            GLib.idle_add(lambda: self.bus.emit('task-updated'))

        self.task_service.run_in_thread(None, do_move)
        return True

class KanbanTaskRow(Gtk.ListBoxRow):
    def __init__(self, task, task_service, board_service, profile_service, root_window, is_blocked=False):
        super().__init__()
        self.task = task
        self.task_service = task_service
        self.board_service = board_service
        self.profile_service = profile_service
        self.root_window = root_window
        
        self.set_activatable(False)
        self.add_css_class("kanban-card-row")
        
        self.card_widget = KanbanCardWidget()
        self.card_widget.bind(task, is_blocked)
        self.set_child(self.card_widget)
        
        self._setup_drag_source()
        self._setup_click()
        self._setup_context_menu()

    def _setup_drag_source(self):
        source = Gtk.DragSource()
        source.set_actions(Gdk.DragAction.MOVE)
        source.connect("prepare", self._on_drag_prepare)
        self.add_controller(source)

    def _on_drag_prepare(self, source, x, y):
        return Gdk.ContentProvider.new_for_value(self.task.id)

    def _setup_click(self):
        click = Gtk.GestureClick()
        click.connect("released", self._on_click)
        self.add_controller(click)

    def _setup_context_menu(self):
        # Right click controller
        right_click = Gtk.GestureClick()
        right_click.set_button(3)
        right_click.connect("pressed", self._on_right_click)
        self.add_controller(right_click)

    def _on_click(self, gesture, n_press, x, y):
        # Left click opens editor
        if gesture.get_current_button() == 1 and self.root_window:
            dialog = TaskEditorDialog(
                self.root_window, 
                self.task_service, 
                self.board_service, 
                self.profile_service, 
                self.task
            )
            dialog.present()

    def _on_right_click(self, gesture, n, x, y):
        pop = Gtk.Popover()
        pop.set_parent(self)
        
        box = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=2)
        box.set_margin_top(6); box.set_margin_bottom(6)
        
        # Toggle Complete
        is_done = self.task.status == Status.DONE
        lbl_status = "Mark Incomplete" if is_done else "Mark Complete"
        btn_status = Gtk.Button(label=lbl_status, css_classes=["flat"])
        btn_status.connect("clicked", lambda b: self._toggle_status(pop))
        box.append(btn_status)
        
        # Delete
        btn_del = Gtk.Button(label="Delete Task", css_classes=["flat", "destructive-action"])
        btn_del.connect("clicked", lambda b: self._delete_task(pop))
        box.append(btn_del)
        
        pop.set_child(box)
        
        # Position menu at click
        rect = Gdk.Rectangle()
        rect.x = int(x); rect.y = int(y); rect.width = 1; rect.height = 1
        pop.set_pointing_to(rect)
        pop.popup()

    def _toggle_status(self, pop):
        pop.popdown()
        new_status = Status.TODO if self.task.status == Status.DONE else Status.DONE
        self.task.status = new_status
        if new_status == Status.DONE:
            from datetime import datetime
            self.task.completed_at = datetime.now()
        else:
            self.task.completed_at = None
            
        self.task_service.run_in_thread(
            lambda r: EventBus.get_default().emit('task-updated'),
            self.task_service.update_task,
            self.task
        )

    def _delete_task(self, pop):
        pop.popdown()
        self.task_service.run_in_thread(
            lambda r: (EventBus.get_default().emit('task-deleted'), EventBus.get_default().emit('task-updated')),
            self.task_service.delete_task,
            self.task.id
        )
