# rush/ui/widgets/task_row.py
import gi
gi.require_version('Gtk', '4.0')
from gi.repository import Gtk, GLib

from rush.data.models import Status
from rush.utils.event_bus import EventBus

class TaskRowWidget(Gtk.Box):
    def __init__(self, task):
        super().__init__(orientation=Gtk.Orientation.HORIZONTAL, spacing=12)
        self.set_margin_top(10); self.set_margin_bottom(10)
        self.set_margin_start(16); self.set_margin_end(16)
        
        self.task = task
        self.bus = EventBus.get_default()
        
        # Checkbox
        self.check = Gtk.CheckButton()
        self.check.set_valign(Gtk.Align.CENTER)
        self.check.connect("toggled", self._on_toggled)
        self.append(self.check)
        
        # Content Box
        vbox = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=4)
        vbox.set_valign(Gtk.Align.CENTER)
        self.append(vbox)
        
        # Title Row
        self.lbl_title = Gtk.Label(xalign=0, css_classes=["body"])
        self.lbl_title.set_ellipsize(3) # End ellipsize
        vbox.append(self.lbl_title)
        
        # Meta Row (Badges + Tags)
        self.meta_box = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=6)
        vbox.append(self.meta_box)
        
        # Priority Badge (Placeholder, set in bind)
        self.lbl_prio = Gtk.Label()
        self.meta_box.append(self.lbl_prio)
        
        # Tags Label
        self.lbl_tags = Gtk.Label(css_classes=["caption", "dim-label"])
        self.meta_box.append(self.lbl_tags)
        
        # Due Date (Far Right)
        self.lbl_due = Gtk.Label(css_classes=["caption"])
        self.lbl_due.set_halign(Gtk.Align.END)
        self.lbl_due.set_hexpand(True)
        self.lbl_due.set_valign(Gtk.Align.CENTER)
        self.append(self.lbl_due)
        
        if task: self.bind(task)

    def bind(self, task):
        self.task = task
        self.lbl_title.set_label(task.title)
        
        # Checkbox State
        self.check.handler_block_by_func(self._on_toggled)
        is_done = task.status == Status.DONE
        self.check.set_active(is_done)
        self.check.handler_unblock_by_func(self._on_toggled)
        
        self._update_style(is_done)
        
        # 1. Vibrant Priority Badge
        prio_val = task.priority.value
        prio_name = task.priority.name.title()
        self.lbl_prio.set_label(prio_name)
        
        # Clear old classes
        self.lbl_prio.remove_css_class("prio-high")
        self.lbl_prio.remove_css_class("prio-medium")
        self.lbl_prio.remove_css_class("prio-low")
        
        if prio_val >= 3:
            self.lbl_prio.add_css_class("prio-high")
        elif prio_val == 2:
            self.lbl_prio.add_css_class("prio-medium")
        else:
            self.lbl_prio.add_css_class("prio-low")

        # 2. Tags
        if task.tags:
            self.lbl_tags.set_label("• " + ", ".join(task.tags))
            self.lbl_tags.set_visible(True)
        else:
            self.lbl_tags.set_visible(False)
        
        # 3. Due Date styling
        if task.due_date:
            self.lbl_due.set_label(task.due_date.strftime("%b %d"))
            from datetime import date
            if task.due_date < date.today() and not is_done:
                self.lbl_due.add_css_class("error") # Adwaita error color
                self.lbl_due.add_css_class("title-4") # Make overdue bold
            else:
                self.lbl_due.remove_css_class("error")
                self.lbl_due.remove_css_class("title-4")
                self.lbl_due.add_css_class("dim-label")
        else:
            self.lbl_due.set_label("")

    def _update_style(self, is_done):
        if is_done:
            self.add_css_class("task-completed")
        else:
            self.remove_css_class("task-completed")

    def _on_toggled(self, btn):
        if not self.task: return
        is_done = btn.get_active()
        self._update_style(is_done)
        
        # Robust update via window look-up
        root = self.get_root()
        if hasattr(root, 'task_service'):
            service = root.task_service
            
            new_status = Status.DONE if is_done else Status.TODO
            self.task.status = new_status
            if is_done:
                from datetime import datetime
                self.task.completed_at = datetime.now()
            else:
                self.task.completed_at = None
                
            service.run_in_thread(
                lambda r: self.bus.emit('task-updated'),
                service.update_task,
                self.task
            )
