# rush/ui/windows/main_window.py
import gi
import os
gi.require_version('Gtk', '4.0')
gi.require_version('Adw', '1')
from gi.repository import Gtk, Adw, Gio, Gdk, GLib

from rush.ui.views.dashboard_view import DashboardView
from rush.ui.views.task_list_view import TaskListView
from rush.ui.views.kanban_view import KanbanView
from rush.ui.views.roadmap_view import RoadmapView
from rush.ui.views.flow_view import FlowView
from rush.ui.views.stats_view import StatsView
from rush.ui.views.calendar_view import CalendarView
from rush.ui.views.graph_view import GraphView 

from rush.ui.dialogs.task_editor import TaskEditorDialog
from rush.ui.dialogs.settings import SettingsDialog
from rush.ui.dialogs.search_dialog import SearchDialog

from rush.utils.event_bus import EventBus

class MainWindow(Adw.ApplicationWindow):
    def __init__(self, app, task_service, board_service, profile_service, flow_service):
        super().__init__(application=app, title="Rush")
        self.task_service = task_service
        self.board_service = board_service
        self.profile_service = profile_service
        self.flow_service = flow_service
        self.bus = EventBus.get_default()
        
        self._setup_ui()
        self.bus.connect('profile-updated', self._on_profile_updated)

    def _setup_ui(self):
        self.set_default_size(1200, 800)

        self.split_view = Adw.OverlaySplitView()
        self.set_content(self.split_view)

        # --- Sidebar ---
        self.sidebar_page = Adw.NavigationPage(title="Rush", tag="sidebar")
        
        sidebar_box = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=12)
        sidebar_box.set_margin_top(12); sidebar_box.set_margin_bottom(12)
        sidebar_box.set_margin_start(12); sidebar_box.set_margin_end(12)
        
        # 1. Profile
        self.profile_row = Adw.ActionRow()
        self.profile_row.set_activatable(True)
        self.profile_row.connect("activated", self._on_settings)
        self.profile_row.add_css_class("profile-card") 
        
        self.avatar_widget = Adw.Avatar(size=48, show_initials=True)
        self.profile_row.add_prefix(self.avatar_widget)
        self.update_sidebar_profile()
        sidebar_box.append(self.profile_row)
        
        # 2. Mode Selector
        mode_box = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=8)
        mode_box.add_css_class("card")
        mode_box.set_size_request(-1, 40)
        
        icon = Gtk.Image(icon_name="system-switch-user-symbolic")
        icon.set_margin_start(12)
        mode_box.append(icon)
        
        self.btn_mode = Gtk.MenuButton()
        self.btn_mode.set_label("Personal")
        self.btn_mode.set_hexpand(True)
        self.btn_mode.set_halign(Gtk.Align.START)
        self.btn_mode.add_css_class("flat")
        self.btn_mode.set_popover(self._create_mode_popover())
        mode_box.append(self.btn_mode)
        
        sidebar_box.append(mode_box)
        sidebar_box.append(Gtk.Separator(orientation=Gtk.Orientation.HORIZONTAL))
        
        # 3. Quick Add
        btn_quick_add = Gtk.Button(label="New Task", icon_name="list-add-symbolic")
        btn_quick_add.add_css_class("suggested-action")
        btn_quick_add.add_css_class("pill")
        btn_quick_add.set_action_name("win.quick-add")
        sidebar_box.append(btn_quick_add)

        # 4. Navigation
        self.stack = Adw.ViewStack()
        
        self.nav_list = Gtk.ListBox()
        self.nav_list.add_css_class("navigation-sidebar")
        self.nav_list.set_selection_mode(Gtk.SelectionMode.SINGLE)
        self.nav_list.connect("row-selected", self._on_nav_selected)
        
        self._add_nav_row("Dashboard", "go-home-symbolic", "dashboard")
        self._add_nav_row("Tasks", "view-list-bullet-symbolic", "tasks")
        self._add_nav_row("Kanban", "view-grid-symbolic", "kanban") 
        self._add_nav_row("Task Graph", "network-workgroup-symbolic", "graph") 
        self._add_nav_row("Calendar", "x-office-calendar-symbolic", "calendar") 
        self._add_nav_row("Roadmap", "view-paged-symbolic", "roadmap") 
        self._add_nav_row("Flow", "media-playback-start-symbolic", "flow")
        self._add_nav_row("Stats", "view-sort-ascending-symbolic", "stats")
        
        sidebar_box.append(self.nav_list)
        
        spacer = Gtk.Box(orientation=Gtk.Orientation.VERTICAL)
        spacer.set_vexpand(True)
        sidebar_box.append(spacer)
        
        # Settings
        btn_settings = Gtk.Button(icon_name="emblem-system-symbolic", label="Settings")
        btn_settings.add_css_class("flat")
        btn_settings.set_action_name("win.settings")
        sidebar_box.append(btn_settings)

        self.sidebar_page.set_child(sidebar_box)
        self.split_view.set_sidebar(self.sidebar_page)

        # --- Content Area ---
        content_nav_view = Adw.NavigationView()
        self.split_view.set_content(content_nav_view)
        
        self.header_bar = Adw.HeaderBar()
        
        btn_search = Gtk.Button(icon_name="system-search-symbolic")
        btn_search.set_action_name("win.command-palette")
        btn_search.set_tooltip_text("Search (Ctrl+K)")
        self.header_bar.pack_start(btn_search)
        
        # Instantiate Views
        self.dashboard_view = DashboardView(self.task_service, self.profile_service, self.flow_service, self.board_service)
        self.task_view = TaskListView(self.task_service, self.profile_service, self.board_service)
        self.kanban_view = KanbanView(self.task_service, self.board_service, self.profile_service)
        self.graph_view = GraphView(self.task_service, self.board_service, self.profile_service)
        self.calendar_view = CalendarView(self.task_service, self.profile_service, self.board_service)
        self.roadmap_view = RoadmapView(self.task_service, self.board_service, self.profile_service)
        self.flow_view = FlowView(self.flow_service, self.task_service)
        self.stats_view = StatsView(self.task_service, self.board_service, self.flow_service)
        
        self.stack.add_named(self.dashboard_view, "dashboard")
        self.stack.add_named(self.task_view, "tasks")
        self.stack.add_named(self.kanban_view, "kanban")
        self.stack.add_named(self.graph_view, "graph") 
        self.stack.add_named(self.calendar_view, "calendar")
        self.stack.add_named(self.roadmap_view, "roadmap")
        self.stack.add_named(self.flow_view, "flow")
        self.stack.add_named(self.stats_view, "stats")
        
        toolbar_view = Adw.ToolbarView()
        toolbar_view.add_top_bar(self.header_bar)
        toolbar_view.set_content(self.stack)
        
        page_content = Adw.NavigationPage(title="Dashboard", tag="content")
        page_content.set_child(toolbar_view)
        content_nav_view.add(page_content)

        self._setup_actions()
        
        # Select first row
        self.nav_list.select_row(self.nav_list.get_row_at_index(0))
        
        prof = self.profile_service.get_user_profile()
        self.btn_mode.set_label(prof.get('current_mode', 'Personal'))

    def _create_mode_popover(self):
        pop = Gtk.Popover()
        box = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=0)
        box.set_margin_top(6); box.set_margin_bottom(6)
        
        modes = [
            ("Personal", "user-available-symbolic"),
            ("Work", "folder-documents-symbolic"),
            ("Study", "accessories-dictionary-symbolic"),
            ("Creative", "applications-graphics-symbolic")
        ]
        
        for m, icon in modes:
            btn = Gtk.Button()
            btn.add_css_class("flat")
            
            hbox = Gtk.Box(spacing=12)
            hbox.set_margin_start(12); hbox.set_margin_end(12)
            hbox.set_margin_top(8); hbox.set_margin_bottom(8)
            
            img = Gtk.Image(icon_name=icon)
            lbl = Gtk.Label(label=m, xalign=0, hexpand=True)
            
            hbox.append(img)
            hbox.append(lbl)
            btn.set_child(hbox)
            
            btn.connect("clicked", self._on_mode_selected, m)
            box.append(btn)
            
        pop.set_child(box)
        return pop

    def _on_mode_selected(self, btn, mode):
        self.btn_mode.set_label(mode)
        self.btn_mode.get_popover().popdown()
        self.profile_service.update_mode(mode)

    def _on_profile_updated(self, bus, *args):
        self.update_sidebar_profile()
        prof = self.profile_service.get_user_profile()
        self.btn_mode.set_label(prof.get('current_mode', 'Personal'))

    def update_sidebar_profile(self):
        profile = self.profile_service.get_user_profile()
        name = profile.get('name') or 'User'
        bio = profile.get('bio') or ""
        avatar_path = profile.get('avatar_path')

        safe_name = GLib.markup_escape_text(name)
        safe_bio = GLib.markup_escape_text(bio)
        
        self.profile_row.set_title(safe_name)
        
        if safe_bio:
            short_bio = safe_bio[:35] + "..." if len(safe_bio) > 35 else safe_bio
            self.profile_row.set_subtitle(short_bio)
        else:
            self.profile_row.set_subtitle("")
        
        self.avatar_widget.set_text(safe_name)
        if avatar_path and os.path.exists(avatar_path):
            try:
                texture = Gdk.Texture.new_from_filename(avatar_path)
                self.avatar_widget.set_custom_image(texture)
            except: pass
        else:
            self.avatar_widget.set_custom_image(None)

    def _add_nav_row(self, title, icon_name, view_name):
        row = Gtk.ListBoxRow()
        row.view_name = view_name
        box = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=12)
        box.set_margin_top(10); box.set_margin_bottom(10)
        box.set_margin_start(12); box.set_margin_end(12)
        
        icon = Gtk.Image(icon_name=icon_name)
        lbl = Gtk.Label(label=title)
        
        box.append(icon)
        box.append(lbl)
        row.set_child(box)
        self.nav_list.append(row)

    def _on_nav_selected(self, listbox, row):
        if row: 
            self.stack.set_visible_child_name(row.view_name)
            child = row.get_child() 
            if child:
                lbl = child.get_last_child()
                title_widget = Adw.WindowTitle(title=lbl.get_label())
                self.header_bar.set_title_widget(title_widget)

    def _setup_actions(self):
        act_add = Gio.SimpleAction.new("quick-add", None)
        act_add.connect("activate", self._on_quick_add)
        self.add_action(act_add)
        
        act_set = Gio.SimpleAction.new("settings", None)
        act_set.connect("activate", self._on_settings)
        self.add_action(act_set)
        
        act_cmd = Gio.SimpleAction.new("command-palette", None)
        act_cmd.connect("activate", self._on_command_palette)
        self.add_action(act_cmd)
        
        app = self.get_application()
        if app: 
            app.set_accels_for_action("win.quick-add", ["<Control>n"])
            app.set_accels_for_action("win.command-palette", ["<Control>k"])

    def _on_quick_add(self, action, param):
        # Pass all required services to the Dialog
        dialog = TaskEditorDialog(self, self.task_service, self.board_service, self.profile_service, None)
        dialog.present()

    def _on_settings(self, *args):
        dialog = SettingsDialog(self, self.task_service, self.profile_service, self.board_service)
        dialog.present()
        
    def _on_command_palette(self, action, param):
        dialog = SearchDialog(self, self.task_service, self)
        dialog.present()
