# rush/utils/formatting.py
import re

def truncate_text(text: str, limit: int = 50, ellipsis: str = "...") -> str:
    """
    Cuts text to a specific length, respecting word boundaries if possible.
    """
    if not text:
        return ""
        
    text = text.strip()
    if len(text) <= limit:
        return text
        
    # Cut to limit
    cut = text[:limit]
    
    # Try to find the last space to avoid cutting a word in half
    last_space = cut.rfind(' ')
    if last_space > limit * 0.8: # Only if space is near the end
        cut = cut[:last_space]
        
    return cut + ellipsis

def strip_markdown(text: str) -> str:
    """
    Removes basic Markdown syntax to create a plain text preview.
    (Removes **, __, #, etc.)
    """
    if not text:
        return ""
    
    # Remove headers (# Title)
    text = re.sub(r'^#+\s+', '', text, flags=re.MULTILINE)
    # Remove bold/italic (**text**, *text*)
    text = re.sub(r'\*\*|__|\*|_', '', text)
    # Remove links [text](url) -> text
    text = re.sub(r'\[([^\]]+)\]\([^\)]+\)', r'\1', text)
    # Remove code blocks `code`
    text = re.sub(r'`', '', text)
    # Remove lists (- item)
    text = re.sub(r'^\s*[-*+]\s+', '', text, flags=re.MULTILINE)
    
    return text.strip()

def format_currency(amount: float, currency_symbol: str = "$") -> str:
    """Format float as currency string."""
    return f"{currency_symbol}{amount:,.2f}"
