# rush/utils/seeder.py
import sqlite3
import uuid
import random
import json
from datetime import datetime, timedelta
from rush.data.db import DatabaseManager

# --- CONFIGURATION ---
START_DATE = datetime(2025, 12, 25)
END_DATE = datetime(2026, 3, 26)
TOTAL_DAYS = (END_DATE - START_DATE).days
TARGET_TASKS = 2200  # High volume target

# --- PROFILE DATA ---
PROFILE_DATA = {
    "name": "Manzoor Ahmed",
    "bio": "Tech Creator, Educator & Author. Dedicated to learning, building, and sharing knowledge. Creator of Laxer OS & Author of 'Muhaar'.",
    "work_hours_start": "08:00",
    "work_hours_end": "18:00",
    "weekly_goal": 45,
    "theme_pref": "dark",
    "current_mode": "Work"
}

# --- PROJECT CONTEXTS ---
PROJECTS = [
    {
        "name": "Laxer OS",
        "mode": "Work",
        "tags": ["Linux", "Dev", "System", "Arch"],
        "tasks": [
            "Update Arch mirrorlist", "Fix Calamares UEFI crash", "Package kernel 6.13",
            "Optimize ZSH config", "Debug Wayland compositor", "Update Pacman hooks",
            "Design bootloader theme", "Test ISO on ThinkPad", "Push AUR updates",
            "Write post-install docs", "Fix audio latency", "Review shell PRs",
            "Compiz window decoration fix", "Polybar config tweak", "Rofi theme update"
        ]
    },
    {
        "name": "Muhaar (The Reins)",
        "mode": "Creative",
        "tags": ["Writing", "Book", "Philosophy", "Deep Work"],
        "tasks": [
            "Draft 'Illusion of Control'", "Proofread 'Behavioral Layers'",
            "Sketch cover art", "Contact publisher", "Format for Kindle",
            "Blog: Self-awareness", "Research philosophy refs", "Edit Introduction",
            "Review reader feedback", "Finalize back cover", "Marketing strategy",
            "Translate Ch.3 to Urdu", "Design social banners", "Record audiobook snippet"
        ]
    },
    {
        "name": "Rush App ⚡",
        "mode": "Work",
        "tags": ["Python", "GTK4", "Gnome", "Productivity"],
        "tasks": [
            "Implement FTS5 search", "Fix Graph clipping", "Refactor Dark Mode CSS",
            "Add Kanban DND", "Debug SQLite threads", "Design empty states",
            "Write RPM spec", "Test Flatpak sandbox", "Optimize Flow Timer",
            "Add Context Menus", "Update README", "Release v1.1.0",
            "Fix window geometry save", "Refactor Service layer", "Add unit tests"
        ]
    },
    {
        "name": "Design & Assets",
        "mode": "Creative",
        "tags": ["Design", "Icons", "SVG", "Blender"],
        "tasks": [
            "Design 'folder-home' icon", "Fix 48px SVG scaling", "YouTube thumbnail #42",
            "Export Zakoota icons", "Redesign website header", "Dark theme palette",
            "Client logo sketches", "Render 3D intro", "Figma component cleanup",
            "Update Behance", "Create broadcast overlay", "Vectorize logo"
        ]
    },
    {
        "name": "Personal & Learning",
        "mode": "Personal",
        "tags": ["Life", "Craft", "Reading", "Health"],
        "tasks": [
            "Buy leather dye", "Stitch wallet prototype", "Sharpen tools",
            "Pattern for notebook", "Burnish belt edges", "Order hardware",
            "Watch saddle stitch tutorial", "Clean workspace", "Leg Day",
            "Read 20 pages", "Grocery run", "Pay internet bill",
            "Call parents", "Organize bookshelf", "Morning run"
        ]
    }
]

def seed_data():
    db = DatabaseManager()
    conn = db.get_connection()
    
    print(f"🌱 Seeding Rush for {PROFILE_DATA['name']}...")
    print(f"📅 Range: {START_DATE.date()} to {END_DATE.date()} (~{TOTAL_DAYS} days)")

    try:
        with conn:
            # 1. Clear Old Data
            print("🧹 Clearing existing data...")
            conn.execute("DELETE FROM tasks")
            conn.execute("DELETE FROM boards")
            conn.execute("DELETE FROM columns")
            conn.execute("DELETE FROM task_dependencies")
            conn.execute("DELETE FROM flow_sessions")
            
            # 2. Populate Profile
            print("👤 Updating User Profile...")
            conn.execute("""
                UPDATE user_profile 
                SET name = ?, bio = ?, work_hours_start = ?, work_hours_end = ?, 
                    weekly_goal = ?, theme_pref = ?, current_mode = ?
                WHERE id = 1
            """, (
                PROFILE_DATA["name"], PROFILE_DATA["bio"], 
                PROFILE_DATA["work_hours_start"], PROFILE_DATA["work_hours_end"], 
                PROFILE_DATA["weekly_goal"], PROFILE_DATA["theme_pref"], 
                PROFILE_DATA["current_mode"]
            ))

            # 3. Create Boards
            print("📋 Creating Boards...")
            board_ids = {} 
            col_map = {}   
            
            for proj in PROJECTS:
                bid = str(uuid.uuid4())
                board_ids[proj["name"]] = bid
                conn.execute("INSERT INTO boards (id, name) VALUES (?, ?)", (bid, proj["name"]))
                
                # Columns
                cols = ["Backlog", "To Do", "In Progress", "Review", "Done"]
                c_ids = []
                for i, cname in enumerate(cols):
                    cid = str(uuid.uuid4())
                    c_ids.append(cid)
                    conn.execute("INSERT INTO columns (id, board_id, name, position, color) VALUES (?, ?, ?, ?, ?)", 
                                 (cid, bid, cname, i, "accent"))
                col_map[bid] = c_ids

            # 4. Generate Tasks
            print(f"🚀 Generating ~{TARGET_TASKS} tasks...")
            
            current_date = START_DATE
            task_count = 0
            
            # Calculate daily volume to hit target
            daily_min = (TARGET_TASKS // TOTAL_DAYS) - 5
            daily_max = (TARGET_TASKS // TOTAL_DAYS) + 8
            
            while current_date <= END_DATE:
                # Progress (0.0 to 1.0)
                progress = (current_date - START_DATE).days / TOTAL_DAYS
                
                # Daily volume
                vol = random.randint(daily_min, daily_max)
                
                for _ in range(vol):
                    proj = random.choice(PROJECTS)
                    bid = board_ids[proj["name"]]
                    cols = col_map[bid]
                    
                    # Status Distribution based on time
                    # Past dates = mostly done. Recent dates = mixed.
                    r = random.random()
                    if progress < 0.85: # Older
                        if r > 0.15: 
                            status = "done"; col_idx = 4
                        else: 
                            status = "todo"; col_idx = 0 # Backlog
                    else: # Recent (last 2 weeks)
                        if r > 0.7: status = "done"; col_idx = 4
                        elif r > 0.4: status = "in_progress"; col_idx = 2
                        else: status = "todo"; col_idx = 1
                    
                    tid = str(uuid.uuid4())
                    title = random.choice(proj["tasks"])
                    desc = f"Work on {title} for {proj['name']} project.\n\n- [ ] Initial research\n- [ ] Implementation\n- [ ] Review"
                    
                    # Timestamps
                    created_at = current_date + timedelta(hours=random.randint(8, 20), minutes=random.randint(0, 59))
                    due_date = (created_at + timedelta(days=random.randint(1, 10))).date()
                    
                    completed_at = None
                    if status == "done":
                        completed_at = created_at + timedelta(days=random.randint(0, 4), hours=random.randint(1, 5))
                        if completed_at > datetime.now(): completed_at = datetime.now()

                    # Checklist
                    checklist = [
                        {"text": "Research & Planning", "completed": True},
                        {"text": "Execution", "completed": status == "done"},
                        {"text": "Final Review", "completed": status == "done"}
                    ]
                    
                    # Tags (Project specific + Generic)
                    task_tags = random.sample(proj["tags"], k=min(2, len(proj["tags"])))
                    
                    conn.execute("""
                        INSERT INTO tasks (
                            id, title, description, status, priority, 
                            board_id, column_id, created_at, completed_at, 
                            mode, due_date, estimated_minutes, checklist_json, tags
                        ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
                    """, (
                        tid, title, desc, status, 
                        random.choices([1, 2, 3, 4], weights=[10, 40, 35, 15])[0], # Priority weights
                        bid, cols[col_idx],
                        created_at.isoformat(),
                        completed_at.isoformat() if completed_at else None,
                        proj["mode"],
                        due_date.isoformat(),
                        random.choice([30, 45, 60, 120]),
                        json.dumps(checklist),
                        json.dumps(task_tags) # Store tags as JSON list
                    ))
                    
                    task_count += 1

                    # 5. Generate Flow Sessions
                    if status in ["done", "in_progress"] and random.random() > 0.4:
                        sessions = random.randint(1, 3)
                        for _ in range(sessions):
                            dur = random.choice([1500, 2700, 3600]) # seconds
                            s_start = created_at + timedelta(hours=random.randint(1, 48))
                            if s_start > datetime.now(): s_start = datetime.now() - timedelta(minutes=60)
                            
                            conn.execute("""
                                INSERT INTO flow_sessions (id, task_id, start_time, duration_seconds)
                                VALUES (?, ?, ?, ?)
                            """, (str(uuid.uuid4()), tid, s_start.isoformat(), dur))

                current_date += timedelta(days=1)

            # 6. Link Tasks (Graph View)
            print("🔗 Linking Tasks...")
            for bid in board_ids.values():
                rows = conn.execute("SELECT id FROM tasks WHERE board_id = ?", (bid,)).fetchall()
                tids = [r[0] for r in rows]
                if len(tids) > 5:
                    for _ in range(15): # 15 links per board
                        p, c = random.sample(tids, 2)
                        conn.execute("INSERT OR IGNORE INTO task_dependencies VALUES (?, ?)", (p, c))

        print(f"✅ SUCCESS! Created {task_count} tasks across {len(PROJECTS)} projects.")
        print("Restart Rush to see your populated dashboard.")

    except Exception as e:
        print(f"❌ Error seeding data: {e}")

if __name__ == "__main__":
    seed_data()
